// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __Vec3_h__
#define __Vec3_h__

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Vec3i_h__
#include "Vec3i.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __FloatUtil_h__
#include "FloatUtil.h"
#endif

class String;
class Vec3 {
  // 3-vectors of Float's
  // Inline's make a measurable difference for this class.
  Float values [3];
 public:
  Float length () const;
  Float lengthSquared () const;
  Vec3 makeUnit () const;
  Float & operator[] (int index);
  void copyToPointer (Float *pointer) const;
  void addToPointer (Float *pointer) const;
  const Float & operator[] (int index) const;
  Float * pointer ();
  const Float * pointer () const;
  Vec3 (Float x, Float y, Float z);
  Vec3 (const Vec3i &v);
  Vec3i truncateToVec3i ();
  Vec3 ();
  // Fetch x, y, and z from the pointer p.
  Vec3 (const Float *p);
  // Return this cross v.
  Vec3 cross (const Vec3 &v) const;
  // Find a unit vector normal to this and return it.
  Vec3 anyVectorNormalTo () const;
  operator String () const;
};

class ostream;
ostream & operator <<(ostream &o, const Vec3 &v);


inline Float & Vec3::operator[] (int index) {
  assert (0 <= index && index < 3);
  return values [index];
}

inline const Float & Vec3::operator[] (int index) const {
  assert (0 <= index && index < 3);
  return values [index];
}

inline Float *Vec3::pointer () {
  return &values[0];
}

inline const Float *Vec3::pointer () const {
  return &values[0];
}

inline Vec3::Vec3 () {
  values [0] = values [1] = values [2] = 0.0;
}

inline Vec3::Vec3 (const Float *p) {
  values [0] = p [0];
  values [1] = p [1];
  values [2] = p [2];
}

inline Vec3::Vec3 (Float x, Float y, Float z) {
  values [0] = x;
  values [1] = y;
  values [2] = z;
}

inline void Vec3::copyToPointer (Float *pointer) const {
  pointer [0] = values [0];
  pointer [1] = values [1];
  pointer [2] = values [2];
}

inline void Vec3::addToPointer (Float *pointer) const {
  pointer [0] += values [0];
  pointer [1] += values [1];
  pointer [2] += values [2];
}

inline Vec3 Vec3::cross (const Vec3 &v) const {
  return Vec3 (values [1] * v [2] - v[1] * values [2],
	       values [2] * v [0] - v[2] * values [0],
	       values [0] * v [1] - v[0] * values [1]);
}

inline Vec3 operator- (const Vec3 &v1, const Vec3 &v2) {
  return Vec3 (v1 [0] - v2 [0], v1 [1] - v2 [1], v1 [2] - v2 [2]);
}

inline Vec3 operator/ (const Vec3 &v, Float d) {
  return Vec3 (v [0] / d, v [1] / d, v [2] / d);
}

inline Float operator* (const Vec3 &v1, const Vec3 &v2) {
  return v1 [0] * v2 [0] + v1 [1] * v2 [1] + v1 [2] * v2 [2];
}

inline Float Vec3::lengthSquared () const {
  return *this * *this;
}

inline Float Vec3::length () const {
  return FloatUtil::sqrt (lengthSquared());
}

inline Vec3 Vec3::makeUnit () const {
  Float l = length ();
  assert (l > 0);
  return (*this) / l;
}

inline Vec3 operator* (const Vec3 &v, Float d) {
  return Vec3 (v[0]*d, v[1]*d, v[2]*d);
}

inline Vec3 operator* (Float d, const Vec3 &v) {
  return v * d;
}

inline Vec3 operator+ (const Vec3 &v1, const Vec3 &v2) {
  return Vec3 (v1[0]+v2[0], v1[1]+v2[1], v1[2]+v2[2]);
}

// Compare the first three components. 
inline bool operator<= (const Vec3 &v1, const Vec3 &v2) {
  return v1[0] <= v2[0] && v1[1] <= v2 [1] && v1 [2] <= v2 [2];
}

inline bool operator== (const Vec3 &v1, const Vec3 &v2) {
  return v1[0] == v2[0] && v1[1] == v2 [1] && v1 [2] == v2 [2];
}

inline bool operator!= (const Vec3 &v1, const Vec3 &v2) {
  return ! (v1 == v2);
}

inline Vec3 &operator+= (Vec3 &v1, const Vec3 &v2) {
  v1 = v1 + v2;
  return v1;
}

inline Vec3 &operator-= (Vec3 &v1, const Vec3 &v2) {
  v1 = v1 - v2;
  return v1;
}

inline Vec3 operator- (const Vec3 &v) {
  return Vec3 (-v[0], -v[1], -v[2]);
}

inline Vec3 Vec3::anyVectorNormalTo () const {
  Vec3 result1 = cross (Vec3 (0, 0, 1));
  Vec3 result2 = cross (Vec3 (0, 1, 0));
  // Take the longer one, since this may have been parallel with
  // either (0, 0, 1) or (0, 1, 0).  It can't be parallel with both!
  if (result1.lengthSquared() > result2.lengthSquared ()) {
    return result1.makeUnit ();
  } else {
    return result2.makeUnit ();
  }
}

inline Vec3i Vec3::truncateToVec3i () {
  return Vec3i ((int) values[0], (int) values[1], (int) values[2]);
}

#endif
