// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __Quaternion_h__
#define __Quaternion_h__

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

class Vec3;
class String;
class Matrix4;
class Quaternion {
  // Represents a rotation in 3-space.  My reference is Advanced Animation and
  // Rendering Techniques: Theory and Practice, Alan Watt and Mark Watt, ACM
  // Press, 1992, pages 360 and following.
  Float m_s;
  Vec3 m_v;
  // After a bunch of operations, we want to renormalize.  This counts the
  // operations.
  int m_staleness;
  Float lengthSquared () const;
  Float length () const;
  Quaternion makeUnit () const;
  void check () const;
  // The next one occasionally renormalizes this, in place, and sets the
  // staleness to 0.
  void occasionallyRenormalize ();
public:
  // Default constructor gives no rotation, and the axis is indeterminate.
  Quaternion ();
  Quaternion (Float s, const Vec3 &v);
  // Return the representation of a rotation of angle theta (radians) about the
  // given axis.  The axis can be any vector; if it is zero, then you get a
  // null quaternion.  We will make the axis vector into a unit vector.
  static Quaternion makeRotation (Float theta, const Vec3 &axis);
  // Apply the rotation to a vector.
  Vec3 rotate (const Vec3 &r) const;
  // To do rotation q1 followed by rotation q2, compute the product q2 * q1.
  // Watt implies otherwise in his example on page 381 in the right column;
  // unfortunately for clarity, the two rotations in his example happen to
  // commute.
  Quaternion operator * (const Quaternion &q);
  operator String () const;
  operator Matrix4 () const;
  // Conjugation.  That is, rotate the other way along the same axis by the
  // same angle, which gives the same quaternion and is equivalent to rotating
  // the same way by along the opposite axis by the same angle.
  Quaternion operator~ () const;
};
#endif
