// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __Quaternion_h__
#include "Quaternion.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __FloatUtil_h__
#include "FloatUtil.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __Matrix4_h__
#include "Matrix4.h"
#endif

namespace {
  const int s_renormFrequency = 97;
}

Float Quaternion::lengthSquared () const {
  return m_s * m_s + m_v * m_v;
}

Float Quaternion::length () const {
  return FloatUtil::sqrt (lengthSquared ());
}

Quaternion Quaternion::makeUnit () const {
  Float l = length ();
  return Quaternion (m_s/l, m_v/l);
}

inline void Quaternion::check () const {
  assert (FloatUtil::abs (1.0 - lengthSquared ()) < 0.01);
}

Quaternion::Quaternion ()
  : m_s (1), m_v (0, 0, 0), m_staleness (0)
{
  check ();
}

Quaternion::Quaternion (Float s, const Vec3 &v)
  : m_s (s), m_v (v), m_staleness (0)
{
  check ();
}

// Return the representation of a rotation of angle theta (radians) about the
// given axis.  The axis can be any vector; if it is zero, then you get a
// null quaternion.  We will make the axis vector into a unit vector.
Quaternion Quaternion::makeRotation (Float theta, const Vec3 &axis) {
  Float lengthSquared = axis.lengthSquared();
  if (0.0 == lengthSquared) {
    return Quaternion ();
  } else {
    Float theta2 = theta / 2.0;
    return Quaternion (cos (theta2), sin (theta2) * axis.makeUnit());
  }
}

Vec3 Quaternion::rotate (const Vec3 &r) const {
  check ();
  return ((m_s*m_s - m_v * m_v) * r +
	  2 * m_v * (m_v * r) +
	  2 * m_s * m_v.cross (r));
}

void Quaternion::occasionallyRenormalize () {
  if (m_staleness > s_renormFrequency) {
    m_staleness = 0;
    Float l = length ();
    m_s = m_s / l;
    m_v = m_v / l;
  }
}

Quaternion Quaternion::operator * (const Quaternion &q) {
  check ();
  q.check ();
  Quaternion result = Quaternion (m_s*q.m_s - m_v * q.m_v,
				  m_s * q.m_v + q.m_s * m_v +
				  m_v.cross (q.m_v));
  result.m_staleness = FloatUtil::max (m_staleness, q.m_staleness) + 1;
  result.occasionallyRenormalize ();
  result.check ();
  return result;
}

Quaternion::operator String () const {
  check ();
  Float halfAngle = FloatUtil::acos (m_s);
  Float angle = 2 * halfAngle;
  if (0.0 == halfAngle) {
    return "Zero rotation";
  } else {
    Vec3 vec = m_v.makeUnit ();
    Float adegrees = angle * 180 / FloatUtil::PI;
    return String ("Rotation of ") + adegrees + " degrees about "+vec;
  }
}

Quaternion::operator Matrix4 () const {
  check ();
  Matrix4 m;
  const Float w = m_s;
  const Float x = m_v[0];
  const Float y = m_v[1];
  const Float z = m_v[2];
  const Float x2 = x * x;
  const Float y2 = y * y;
  const Float z2 = z * z;
  m(0,0)=1-2*y2-2*z2; m(0,1)=2*x*y-2*w*z; m(0,2)=2*x*z+2*w*y; m(0,3)=0;
  m(1,0)=2*x*y+2*w*z; m(1,1)=1-2*x2-2*z2; m(1,2)=2*y*z-2*w*x; m(1,3)=0;
  m(2,0)=2*x*z-2*w*y; m(2,1)=2*y*z+2*w*x; m(2,2)=1-2*x2-2*y2; m(2,3)=0;
  m(3,0)=0;           m(3,1)=0;           m(3,2)=0;           m(3,3)=1;
  return m;
}

Quaternion Quaternion::operator~ () const {
  return Quaternion (m_s, -m_v);
}
