// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __Matrix4_h__
#define __Matrix4_h__
// 4 by 4 matrices of Float's.

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

class Matrix4 {
  Float values [16];
 public:
  Float *pointer ();
  const Float *pointer () const;
  // The constructor initializes to zero.
  Matrix4 ();
  // Same indexing scheme used here as in the OpenGL documentation.
  const Float &operator[] (int i) const;
  Float &operator[] (int i);
  void copyToDouble (double dest [16]) const;
  // If error is specified, and the inversion isn't possible, then set *error
  // to true.  If no error happens and error is specified, leave *error alone,
  // so it can be used to accumulate errors.  If error is unspecified, and an
  // error happens, then abort.
  Matrix4 invert (bool *error = 0) const;
  // Next one adds a "1" as the fourth coordinate of the vector, and multiplies
  // that column on the vector on the left by *this, then trims off the unused
  // fourth coordinate and returns the first three as a vector.  This isn't
  // good for perspective, where the fourth coordinate is used.
  Vec3 operator* (const Vec3 &v) const;
  // Function call is subscripting.
  inline Float &operator() (int r, int c) {
    return values [c * 4 + r];
  }
  inline Float operator() (int r, int c) const {
    return values [c * 4 + r];
  }

  // Compute a matrix that will translate by the given amount.
  static Matrix4 translate (const Vec3 &v);
  
  // Compute a matrix that will scale by the given amount.
  static Matrix4 scale (Float s);
  
  // Compute a matrix that will flip the Z coordinate.
  static Matrix4 flipZ ();
  
  // Multiply matrices.  We transform points by multiplying them on the right,
  // so if you want to do transformation A followed by B, then convert them to
  // matrices matrixA and matrixB, and the resulting transformation is
  // matrixB * matrixA.  The order is backwards from what you might expect.
  Matrix4 operator* (const Matrix4 &m2) const;

};

// Much of the implementation is here so we can inline.

#ifndef __myassert_h__
#include "myassert.h"
#endif

inline Matrix4::Matrix4 () {
  for (int i = 0; i < 16; i++) {
    values [i] = 0.0;
  }
}

inline Float *Matrix4::pointer () {
  return values;
}

inline const Float *Matrix4::pointer () const {
  return values;
}

inline const Float & Matrix4::operator[] (int i) const {
  assert (0 <= i);
  assert (i < 16);
  return values [i];
}

inline Float & Matrix4::operator[] (int i) {
  assert (0 <= i);
  assert (i < 16);
  return values [i];
}

inline Vec3 Matrix4::operator* (const Vec3 &v) const {
  // We do this operation once per object in the scene graph.
  Vec3 result;
  const Matrix4 &m = *this;
  // Unrolled.
  result [0] = m(0, 0) * v[0] + m(0, 1) * v [1] + m(0, 2) * v[2] + m(0,3);
  result [1] = m(1, 0) * v[0] + m(1, 1) * v [1] + m(1, 2) * v[2] + m(1,3);
  result [2] = m(2, 0) * v[0] + m(2, 1) * v [1] + m(2, 2) * v[2] + m(2,3);
  return result;
}

class ostream;

ostream &operator<< (ostream &o, const Matrix4 &m);

#endif
