// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman, Copyright 1995-1999 Brian Paul 
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// Tim Freeman can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//
// According to the README file for Mesa 3.1, Brian Paul can be reached at:
//
// Brian Paul
// Avid Technology
// 1925 Andover St.
// Tewksbury, MA  01876
//
// brian_paul@mesa3d.org

#include "Matrix4.h"

// For ostream, endl
#ifndef __iostream_h__
#include <iostream.h>
#define __iostream_h__
#endif

ostream &operator<< (ostream &o, const Matrix4 &m) {
  for (int i = 0; i < 4; i++) {
    for (int j = i; j < 16; j+=4) {
      o << m[j] << ", ";
    }
    o << endl;
  }
  return o;
}

void Matrix4::copyToDouble (double dest [16]) const {
  for (int i = 0; i < 16; i++) {
    dest [i] = values [i];
  }
}

namespace {
  inline void SWAP_ROWS (Float *&a, Float *&b) {
    Float *_tmp = a; (a)=(b); (b)=_tmp;
  }
  Matrix4 doError (bool *error) {
    if (error) {
      *error = true;
      return Matrix4 ();
    } else {
      cerr << "Attempt to invert singular matrix." << endl;
      abort ();
    }
  }
}

Matrix4 Matrix4::invert (bool *error = 0) const {
  // Matrix inversion code taken from project.c in src-glu from Mesa 3.1.
  // That code has the GNU Library General Public License (version 2), and is
  // Copyright Brian Paul 1995-1999.  The matrix inversion code itself was
  // contributed by Jacques Leroy jle@star.be.
  Matrix4 out;
  
  const Matrix4 &m = *this;
  /* NB. OpenGL Matrices are COLUMN major. */
  Float wtmp[4][8];
  Float m0, m1, m2, m3, s;
  Float *r0, *r1, *r2, *r3;
  
  r0 = wtmp[0], r1 = wtmp[1], r2 = wtmp[2], r3 = wtmp[3];
  
  r0[0] = m(0,0), r0[1] = m(0,1);
  r0[2] = m(0,2), r0[3] = m(0,3);
  r0[4] = 1.0, r0[5] = r0[6] = r0[7] = 0.0;
    
  r1[0] = m(1,0), r1[1] = m(1,1);
  r1[2] = m(1,2), r1[3] = m(1,3);
  r1[5] = 1.0, r1[4] = r1[6] = r1[7] = 0.0;
    
  r2[0] = m(2,0), r2[1] = m(2,1);
  r2[2] = m(2,2), r2[3] = m(2,3);
  r2[6] = 1.0, r2[4] = r2[5] = r2[7] = 0.0;
    
  r3[0] = m(3,0), r3[1] = m(3,1);
  r3[2] = m(3,2), r3[3] = m(3,3);
  r3[7] = 1.0, r3[4] = r3[5] = r3[6] = 0.0;
  
  /* choose pivot - or die */
  if (fabs(r3[0])>fabs(r2[0])) SWAP_ROWS(r3, r2);
  if (fabs(r2[0])>fabs(r1[0])) SWAP_ROWS(r2, r1);
  if (fabs(r1[0])>fabs(r0[0])) SWAP_ROWS(r1, r0);
  if (0.0 == r0[0]) return doError (error);
  
  /* eliminate first variable     */
  m1 = r1[0]/r0[0]; m2 = r2[0]/r0[0]; m3 = r3[0]/r0[0];
  s = r0[1]; r1[1] -= m1 * s; r2[1] -= m2 * s; r3[1] -= m3 * s;
  s = r0[2]; r1[2] -= m1 * s; r2[2] -= m2 * s; r3[2] -= m3 * s;
  s = r0[3]; r1[3] -= m1 * s; r2[3] -= m2 * s; r3[3] -= m3 * s;
  s = r0[4];
  if (s != 0.0) { r1[4] -= m1 * s; r2[4] -= m2 * s; r3[4] -= m3 * s; }
  s = r0[5];
  if (s != 0.0) { r1[5] -= m1 * s; r2[5] -= m2 * s; r3[5] -= m3 * s; }
  s = r0[6];
  if (s != 0.0) { r1[6] -= m1 * s; r2[6] -= m2 * s; r3[6] -= m3 * s; }
  s = r0[7];
  if (s != 0.0) { r1[7] -= m1 * s; r2[7] -= m2 * s; r3[7] -= m3 * s; }
  
  /* choose pivot - or die */
  if (fabs(r3[1])>fabs(r2[1])) SWAP_ROWS(r3, r2);
  if (fabs(r2[1])>fabs(r1[1])) SWAP_ROWS(r2, r1);
  if (0.0 == r1[1]) return doError (error);
  
  /* eliminate second variable */
  m2 = r2[1]/r1[1]; m3 = r3[1]/r1[1];
  r2[2] -= m2 * r1[2]; r3[2] -= m3 * r1[2];
  r2[3] -= m2 * r1[3]; r3[3] -= m3 * r1[3];
  s = r1[4]; if (0.0 != s) { r2[4] -= m2 * s; r3[4] -= m3 * s; }
  s = r1[5]; if (0.0 != s) { r2[5] -= m2 * s; r3[5] -= m3 * s; }
  s = r1[6]; if (0.0 != s) { r2[6] -= m2 * s; r3[6] -= m3 * s; }
  s = r1[7]; if (0.0 != s) { r2[7] -= m2 * s; r3[7] -= m3 * s; }
  
  /* choose pivot - or die */
  if (fabs(r3[2])>fabs(r2[2])) SWAP_ROWS(r3, r2);
  if (0.0 == r2[2]) return doError (error);
  
  /* eliminate third variable */
  m3 = r3[2]/r2[2];
  r3[3] -= m3 * r2[3], r3[4] -= m3 * r2[4];
  r3[5] -= m3 * r2[5], r3[6] -= m3 * r2[6];
  r3[7] -= m3 * r2[7];
  
  /* last check */
  if (0.0 == r3[3]) return doError (error);
  
  s = 1.0/r3[3];              /* now back substitute row 3 */
  r3[4] *= s; r3[5] *= s; r3[6] *= s; r3[7] *= s;
  
  m2 = r2[3];                 /* now back substitute row 2 */
  s  = 1.0/r2[2];
  r2[4] = s * (r2[4] - r3[4] * m2), r2[5] = s * (r2[5] - r3[5] * m2);
  r2[6] = s * (r2[6] - r3[6] * m2), r2[7] = s * (r2[7] - r3[7] * m2);
  m1 = r1[3];
  r1[4] -= r3[4] * m1, r1[5] -= r3[5] * m1;
  r1[6] -= r3[6] * m1, r1[7] -= r3[7] * m1;
  m0 = r0[3];
  r0[4] -= r3[4] * m0, r0[5] -= r3[5] * m0;
  r0[6] -= r3[6] * m0, r0[7] -= r3[7] * m0;
  
  m1 = r1[2];                 /* now back substitute row 1 */
  s  = 1.0/r1[1];
  r1[4] = s * (r1[4] - r2[4] * m1), r1[5] = s * (r1[5] - r2[5] * m1);
  r1[6] = s * (r1[6] - r2[6] * m1), r1[7] = s * (r1[7] - r2[7] * m1);
  m0 = r0[2];
  r0[4] -= r2[4] * m0, r0[5] -= r2[5] * m0;
  r0[6] -= r2[6] * m0, r0[7] -= r2[7] * m0;
  
  m0 = r0[1];                 /* now back substitute row 0 */
  s  = 1.0/r0[0];
  r0[4] = s * (r0[4] - r1[4] * m0), r0[5] = s * (r0[5] - r1[5] * m0);
  r0[6] = s * (r0[6] - r1[6] * m0), r0[7] = s * (r0[7] - r1[7] * m0);
  
  out(0,0) = r0[4]; out(0,1) = r0[5];
  out(0,2) = r0[6]; out(0,3) = r0[7];
  out(1,0) = r1[4]; out(1,1) = r1[5];
  out(1,2) = r1[6]; out(1,3) = r1[7];
  out(2,0) = r2[4]; out(2,1) = r2[5];
  out(2,2) = r2[6]; out(2,3) = r2[7];
  out(3,0) = r3[4]; out(3,1) = r3[5];
  out(3,2) = r3[6]; out(3,3) = r3[7]; 
  
  return out;
}

Matrix4 Matrix4::translate (const Vec3 &v) {
  Matrix4 result;
  // The constructor initializes to zero for us.
  for (int i = 0; i < 3; i++) {
    result (i, i) = 1;
    result (i, 3) = v [i];
  }
  result (3, 3) = 1;
  return result;
}
  
Matrix4 Matrix4::scale (Float s) {
  Matrix4 result;
  for (int i = 0; i < 3; i++) {
    result (i, i) = s;
  }
  result (3, 3) = 1;
  return result;
}
  
Matrix4 Matrix4::flipZ () {
  Matrix4 result;
  for (int i = 0; i < 4; i++) {
    result (i, i) = 1;
  }
  result (2, 2) = -1;
  return result;
}
  
Matrix4 Matrix4::operator* (const Matrix4 &m2) const {
  const Matrix4 &m1 = *this;
  Matrix4 result;
  for (int i = 0; i < 4; i++) {
    for (int j = 0; j < 4; j++) {
      for (int k = 0; k < 4; k++) {
	result (i, j) += m1 (i, k) * m2 (k, j);
      }
    }
  }
  return result;
}

