// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __XKeyboardMouse_h__
#include "XKeyboardMouse.h"
#endif

#ifndef __KeyboardMouseEvent_h__
#include "KeyboardMouseEvent.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __X11_keysym_h__
#include <X11/keysym.h>
#define __X11_keysym_h__
#endif

#ifndef __FloatUtil_h__
#include "FloatUtil.h"
#endif

#ifndef __iostream_h__
#include <iostream.h>
#define __iostream_h__
#endif

#ifndef __IMWheel_h__
#include "IMWheel.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

struct XKeyboardMouse::XKeyboardMouseData {
  int m_shiftButtonCount, m_controlButtonCount, m_metaButtonCount;
  bool m_haveEvent;
  XKeyboardMouseData ()
    : m_shiftButtonCount (0), m_controlButtonCount (0), m_metaButtonCount (0),
      m_haveEvent (false)
  {}
  // Update the shift, control, and meta button counts from the state field of
  // the Xevent.
  void updateFromState (unsigned int state) {
    if (state & ShiftMask) {
      if (0 >= m_shiftButtonCount) {
	// m_shiftButtonCount might be 2 if we observed both shift buttons
	// going down.
	// FIXME Why do we bother counting this?
	// The mask tells us all we want to know.
	m_shiftButtonCount = 1;
      }
    } else {
      m_shiftButtonCount = 0;
    }
    if (state & ControlMask) {
      if (0 >= m_controlButtonCount) {
	m_controlButtonCount = 1;
      }
    } else {
      m_controlButtonCount = 0;
    }
    if (state & Mod1Mask) {
      if (0 >= m_metaButtonCount) {
	m_metaButtonCount = 1;
      }
    } else {
      m_metaButtonCount = 0;
    }
  }
};

XKeyboardMouse::XKeyboardMouse ()
  : KeyboardMouse (),
    m_data (NEW (XKeyboardMouseData  ()))
{}

XKeyboardMouse::~XKeyboardMouse ()
{
  assert (m_data);
  delete m_data;
  m_data = 0;
}

bool XKeyboardMouse::insertXEvent (const XEvent *xevent, int height) {
  SP<KeyboardMouseEvent> kme = getKeyboardMouseEvent ();
  switch (xevent->type) {
  case KeyPress:
  case KeyRelease:
    {
      m_data->updateFromState (xevent->xkey.state);
      // Need ks to be local, otherwise we get gripes about jumping around
      // it during initialization.
      // I think I need const_cast because XLookupKeysym is misdeclared.
      const KeySym ks =
	XLookupKeysym (const_cast <XKeyEvent *> (&(xevent->xkey)), 0);
      // Want down to be local, so I don't use it uninitialized in other
      // branches of the case. 
      const bool down = KeyPress == xevent->type;
      kme->setX (xevent->xkey.x);
      kme->setY (height - xevent->xkey.y - 1);
      // Looking at /usr/include/X11/keysymdef.h, for the keysyms we care
      // about, either we think of it as a button, or the keysym is equal to
      // the ascii code.
      switch (ks) {
	// These cases are listed in the same order as the button types in
	// Event.h.
      case XK_Shift_L:
      case XK_Shift_R:
	if (KeyPress == xevent->type) {
	  m_data->m_shiftButtonCount =
	    FloatUtil::min (2, m_data->m_shiftButtonCount + 1);
	} else {
	  m_data->m_shiftButtonCount =
	    FloatUtil::max (0, m_data->m_shiftButtonCount - 1);
	}
	kme->setButton (KeyboardMouseEvent::SHIFT, down);
	break;
      case XK_Control_L:
      case XK_Control_R:
	if (KeyPress == xevent->type) {
	  m_data->m_controlButtonCount =
	    FloatUtil::min (2, m_data->m_controlButtonCount + 1);
	} else {
	  m_data->m_controlButtonCount =
	    FloatUtil::max (0, m_data->m_controlButtonCount - 1);
	}
	kme->setButton (KeyboardMouseEvent::CONTROL, down);
	break;
	// If there had been one more kind of shift key, I would do something
	// general-purpose here. 
      case XK_Meta_L:
      case XK_Meta_R:
      case XK_Alt_L:
      case XK_Alt_R:
	if (KeyPress == xevent->type) {
	  m_data->m_metaButtonCount =
	    FloatUtil::min (4, m_data->m_metaButtonCount + 1);
	} else {
	  m_data->m_metaButtonCount =
	    FloatUtil::max (0, m_data->m_metaButtonCount - 1);
	}
	kme->setButton (KeyboardMouseEvent::META, down);
	break;
      case XK_Home:
	kme->setButton (KeyboardMouseEvent::HOME, down);
	break;
      case XK_Delete:
	kme->setButton (KeyboardMouseEvent::DELETE, down);
	break;
      case XK_End:
	kme->setButton (KeyboardMouseEvent::END, down);
	break;
      case XK_BackSpace:
	kme->setButton (KeyboardMouseEvent::BACKSPACE, down);
	break;
      case XK_Page_Up:
	kme->setButton (KeyboardMouseEvent::PAGEUP, down);
	break;
      case XK_Page_Down:
	kme->setButton (KeyboardMouseEvent::PAGEDOWN, down);
	break;
      case XK_KP_Enter:
      case XK_ISO_Enter:
      case XK_Return:
	kme->setButton (KeyboardMouseEvent::RETURN, down);
	break;
      case XK_KP_Space:
      case XK_space:
	kme->setButton (KeyboardMouseEvent::SPACE, down);
	break;
      case XK_Tab:
      case XK_KP_Tab:
      case XK_ISO_Left_Tab:
	kme->setButton (KeyboardMouseEvent::TAB, down);
	break;
      case XK_Escape:
	kme->setButton (KeyboardMouseEvent::ESCAPE, down);
	break;
      case XK_Left:
	kme->setButton (KeyboardMouseEvent::LEFTARROW, down);
	break;
      case XK_Right:
	kme->setButton (KeyboardMouseEvent::RIGHTARROW, down);
	break;
      case XK_Up:
	kme->setButton (KeyboardMouseEvent::UPARROW, down);
	break;
      case XK_Down:
	kme->setButton (KeyboardMouseEvent::DOWNARROW, down);
	break;
      default:
	char cks = (char) ks;
	if (((int) ks) == cks) {
	  if (kme->isValidButton (cks)) {
	    kme->setButton (cks, down);
	  } else {
	    message (String ("Unrecognized key ") + (int) ks +
		     " in convertToMyEvent");
	    return false;
	  }
	} // end if ks fits into a char.
      } // end switch on the keycode
    } // End block that makes down and ks local.
    break; // end case for KeyPress or KeyRelease.
  case MotionNotify:
    m_data->updateFromState (xevent->xmotion.state);
    kme->setButton (KeyboardMouseEvent::MOTION, true);
    kme->setX (xevent->xmotion.x);
    kme->setY (height - xevent->xmotion.y - 1);
    break;
  case ButtonPress:
  case ButtonRelease:
    m_data->updateFromState (xevent->xbutton.state);
    { // Make these local so we don't get gripes about jumping across the
      // initialization. 
      const int wheelUp = IMWheel::WheelUp + 1;
      const int wheelDown = IMWheel::WheelDown + 1;
      const bool down = ButtonPress == xevent->type;
      kme->setX (xevent->xbutton.x);
      kme->setY (height - xevent->xbutton.y - 1);
      switch (xevent->xbutton.button) {
      case Button1:
	kme->setButton (KeyboardMouseEvent::LEFT_MOUSE, down);
	break;
      case Button2:
	kme->setButton (KeyboardMouseEvent::MIDDLE_MOUSE, down);
	break;
      case Button3:
	kme->setButton (KeyboardMouseEvent::RIGHT_MOUSE, down);
	break;
      case wheelUp:
	kme->setButton (KeyboardMouseEvent::WHEEL_UP, down);
	break;
      case wheelDown:
	kme->setButton (KeyboardMouseEvent::WHEEL_DOWN, down);
	break;
      default:
	message (String ("Bad button ") + (int) xevent->xbutton.button +
		 " in convertToMyEvent");
	kme->setButton (KeyboardMouseEvent::MOTION, true);
	return false;
	break;
      } // end switch on the button
    } // end wheelup and down are local
    break; // end case ButtonPress or ButtonRelease
  default:
    // We are offered all X events and the rest of the code only gets the X
    // events if we don't want them, so this case is reached quite often.
    // Leave everything unchanged.
    return false;
    break;
  } // end switch on the event type
  // Next three merrily (and correctly) overwrite what we just did.
  // They're special because people commonly have multiple copies of
  // these shift keys on their keyboard.
  kme->setButtonDown (KeyboardMouseEvent::SHIFT,
		      0 != m_data->m_shiftButtonCount);
  kme->setButtonDown (KeyboardMouseEvent::META,
		      0 != m_data->m_metaButtonCount);
  kme->setButtonDown (KeyboardMouseEvent::CONTROL,
		      0 != m_data->m_controlButtonCount);
  m_data->m_haveEvent = true;
  return true;
}

bool XKeyboardMouse::readEvent () {
  bool result = m_data->m_haveEvent;
  m_data->m_haveEvent = false;
  return result;
}

int XKeyboardMouse::fdForSelect () const {
  return -1;
}
