// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __SpaceOrbEvent_h__
#include "SpaceOrbEvent.h"
#endif

#ifndef __Quaternion_h__
#include "Quaternion.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __SpaceOrbConfiguration_h__
#include "SpaceOrbConfiguration.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __Event_h__
#include "Event.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

typedef Event::Button Button;

const Button SpaceOrbEvent::A = 'A';
const Button SpaceOrbEvent::B = 'B';
const Button SpaceOrbEvent::C = 'C';
const Button SpaceOrbEvent::D = 'D';
const Button SpaceOrbEvent::E = 'E';
const Button SpaceOrbEvent::F = 'F';
const Button SpaceOrbEvent::MOTION = 'm';
const Button SpaceOrbEvent::RESET = 'r';
namespace {
  const Button FIRST = SpaceOrbEvent::A;
  const Button LAST = SpaceOrbEvent::RESET + 1;
}

struct SpaceOrbEvent::SpaceOrbEventData {
  Quaternion m_rotation;
  Vec3 m_translation;
  SP<SpaceOrbConfiguration> m_conf;
  SpaceOrbEventData ()
    : m_translation (0, 0, 0)
  {}
};

SpaceOrbEvent::SpaceOrbEvent ()
  : Event (LAST),
    m_data (NEW (SpaceOrbEventData ()))
{}

SpaceOrbEvent::~SpaceOrbEvent () {
  assert (m_data);
  delete m_data;
  m_data = 0;
}

Button SpaceOrbEvent::minButton () const {
  return FIRST;
}

Button SpaceOrbEvent::maxButton () const {
  return LAST;
}

Button SpaceOrbEvent::pointerMotionButton () const {
  return MOTION;
}

bool SpaceOrbEvent::isValidButton (const Button ch) const {
  return ((ch >= A && ch <= F) || MOTION == ch || RESET == ch);
}

Vec3 SpaceOrbEvent::getTranslation () const {
  return m_data->m_translation;
}

void SpaceOrbEvent::setTranslation (const Vec3 &v) {
  m_data->m_translation = v;
}

Quaternion SpaceOrbEvent::getRotation () const {
  return m_data->m_rotation;
}

void SpaceOrbEvent::setRotation (const Quaternion &q) {
  m_data->m_rotation = q;
}

String SpaceOrbEvent::unParseButton (Button b) const {
  switch (b) {
  case A: return "A";
  case B: return "B";
  case C: return "C";
  case D: return "D";
  case E: return "E";
  case F: return "F";
  case MOTION: return "Orb motion";
  case RESET: return "Reset";
  default:
    die (String ("Bad button ") + (int) b +" in Space Orb unParseButton");
    // Compile doesn't know that die doesn't return.
    return "Error";
  }
}

String SpaceOrbEvent::unParse () const {
  if (getButton () != pointerMotionButton ()) {
    return unParseButton (getButton()) + " " + (isDown()?"down":"up");
  } else {
    return "Orb moved to rotation "+m_data->m_rotation+" translation "
      + m_data->m_translation;
  }
}
