// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __SpaceOrb_h__
#include "SpaceOrb.h"
#endif

#ifndef __InputDevicesFactory_h__
#include "InputDevicesFactory.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __SimpleConfiguration_h__
#include "SimpleConfiguration.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __Event_h__
#include "Event.h"
#endif

#ifndef __SpaceOrbEvent_h__
#include "SpaceOrbEvent.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __SpaceOrbConfiguration_h__
#include "SpaceOrbConfiguration.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __LinuxJoystick_h__
#include "LinuxJoystick.h"
#endif

#ifndef __JoystickEvent_h__
#include "JoystickEvent.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __Quaternion_h__
#include "Quaternion.h"
#endif

#ifndef __InputDevices_h__
#include "InputDevices.h"
#endif

namespace {
  String spaceOrbName () {
    return "SpaceOrb 360";
  }
  typedef Event::Button Button;
  class LinuxSpaceOrb
    : public SpaceOrb
  {
    SP<LinuxJoystick> m_joystick;
    SP<SpaceOrbEvent> m_event;
    SP<const JoystickEvent> m_jsevent;
    SP<const SpaceOrbConfiguration> m_conf;
  public:
    LinuxSpaceOrb (SP <LinuxJoystick> lj)
      : m_joystick (lj)
    {
      assert (m_joystick->getName () == spaceOrbName ());
      m_event = dynamic_cast <SpaceOrbEvent *> (&*getVariableEvent ());
      assert (m_event);
      m_jsevent =
	dynamic_cast <const JoystickEvent *> (&*m_joystick->getEvent ());
      assert (m_jsevent);
      m_conf = dynamic_cast <const SpaceOrbConfiguration *> (&*getConf ());
      assert (m_conf);
    }
    Float compStick (Float num, Float stick) {
      if (num > stick) {
	return num - stick;
      } else if (num < -stick) {
	return num + stick;
      } else {
	return 0;
      }
    }
    bool readEvent () {
      if (m_joystick->readEvent ()) {
	Button b = m_jsevent->getButton ();
	Button newB;
	switch (b) {
	case 0: case 1: case 2: case 3: case 4: case 5:
	  newB = b + SpaceOrbEvent::A;
	  break;
	case 6:
	  newB = SpaceOrbEvent::RESET;
	  break;
	default:
	  newB = m_event->pointerMotionButton ();
	  if (b == m_jsevent->pointerMotionButton ()) {
	    // Several issues here:
	    // 1. I'll have to experiment to discover which physical ball
	    // motion is translated to which sign direction on which axis.
	    // 2. The user might want a different correspondence between
	    // Sporb motion and coordinate information depending on how he
	    // holds the sporb.  I'll ignore this issue until a user with a
	    // different preference from me complains.
	    // Doing a malloc call here.  Could hoist it into an instance
	    // variable.  I don't think there are enough events for this to
	    // matter.
	    Dynavec <Float> axes;
	    m_jsevent->getAxes (axes);
	    assert (6 == axes.size());
	    {
	      const Float tGain = m_conf->getTranslationalGain ();
	      const Float tStick = m_conf->getTranslationalStick ();
	      Vec3 t;
	      t [0] = compStick (axes [0] * tGain, tStick);
	      t [1] = compStick (axes [1] * tGain, tStick);
	      t [2] = compStick (axes [2] * tGain, tStick);
	      m_event->setTranslation (t);
	    }
	    {
	      const Float rGain = m_conf->getRotationalGain ();
	      const Float rStick = m_conf->getRotationalStick ();
	      Quaternion r;
	      r =
		Quaternion::makeRotation (compStick (axes [3] * rGain, rStick),
					  Vec3 (1, 0, 0)) *
		Quaternion::makeRotation (compStick (axes [4] * rGain, rStick),
					  Vec3 (0, 1, 0)) *
		Quaternion::makeRotation (compStick (axes [5] * rGain, rStick),
					  Vec3 (0, 0, 1));
	      m_event->setRotation (r);
	    }
	  } else {
	    die (String ("Impossible button ") + (int) b +" from SpaceOrb.");
	    // No translation or rotation.
	    m_event->setRotation (Quaternion ());
	    m_event->setTranslation (Vec3 (0,0,0));
	  }
	  break;
	}
	m_event->setButton (newB, m_jsevent->isDown ());
	return true;
      } else {
	return false;
      }
    }
    int fdForSelect () const {
      return m_joystick->fdForSelect ();
    }
  };
  class LSOInputDevices
    : public InputDevices
  {
  public:
    void findDevices (Dynavec <SP <InputDevice> > &inputDevices) {
      inputDevices.extendTo (0);
      Dynavec <String> devNames = LinuxJoystick::joystickDeviceNames ();
      for (int i = 0; i < devNames.size(); i++) {
	SP<LinuxJoystick> lj =
	  LinuxJoystick::openJoystickDevice (devNames [i]);
	if (lj && (spaceOrbName () == lj->getName ())) {
	  inputDevices.push (NEW (LinuxSpaceOrb (lj)));
	}
      }
    }
  };
  class LinuxSpaceOrbFactory
    : public InputDevicesFactory
  {
  public:
    LinuxSpaceOrbFactory ()
      : InputDevicesFactory ("LinuxSpaceOrbFactory")
    {}
    Dynavec <SP <InputDevicesFactory> > inferiorDeviceFactories () const {
      Dynavec <SP <InputDevicesFactory> > result;
      SP<InputDevicesFactory> linuxJoy =
	dynamic_cast <InputDevicesFactory *>
	(&*FactoryTable::load ("InputDevicesFactory", "LinuxJoystickFactory"));
      assert (linuxJoy);
      result.push (linuxJoy);
      return result;
    }
    SP<Configuration> defaultConfiguration () const {
      return NEW (SimpleConfiguration ());
    }
    SP<Configurable> makeIt (SP<Configuration> conf) const {
      (void) conf;
      return NEW (LSOInputDevices ());
    }
  };
  static const bool useless =
  (FactoryTable::store ("InputDevicesFactory", NEW (LinuxSpaceOrbFactory ())),
   true);
};
