// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __LinuxJoystick_h__
#include "LinuxJoystick.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __linux_joystick_h__
#include <linux/joystick.h>
#define __linux_joystick_h__
#endif

#ifndef __stdio_h__
#include <stdio.h>
#define __stdio_h__
#endif

#ifndef __stdlib_h__
#include <stdlib.h>
#define __stdlib_h__
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __JoystickEvent_h__
#include "JoystickEvent.h"
#endif

#ifndef __unistd_h__
#include <unistd.h>
#define __unistd_h__
#endif

// Next 3 required for open, according to the manual entries.
#ifndef __sys_types_h__
#include <sys/types.h>
#define __sys_types_h__
#endif

#ifndef __sys_stat_h__
#include <sys/stat.h>
#define __sys_stat_h__
#endif

#ifndef __fcntl_h__
#include <fcntl.h>
#define __fcntl_h__
#endif

#ifndef __errno_h__
#include <errno.h>
#define __errno_h__
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __sys_ioctl_h__
#include <sys/ioctl.h>
#define __sys_ioctl_h__
#endif

#ifndef __VecUtil_h__
#include "VecUtil.h"
#endif

#ifndef __Joystick_h__
#include "Joystick.h"
#endif

namespace {
  int getAxes (int fd) {
    unsigned char axes = (unsigned char) -1;
    if (0 != ioctl (fd, JSIOCGAXES, &axes)) {
      perror ("getAxes for Linux joystick");
      exit (1);
    }
    assert (axes != (unsigned char) -1);
    return (int) axes;
  }
  int getButtons (int fd) {
    unsigned char buttons = (unsigned char) -1;
    if (0 != ioctl (fd, JSIOCGBUTTONS, &buttons)) {
      perror ("getButtons for Linux joystick");
      exit (1);
    }
    assert (buttons != (unsigned char) -1);
    return (int) buttons;
  }
  int openDeviceCount = 0;
  // This array says whether each joystick is open.  The index into the array
  // is the index into joystickDeviceNames that is the name of the joystick
  // device.  I want the array to be variable-length, so we only have to update
  // one place if the number of possible joysticks changes, but I also had to
  // avoid global variables that point to allocated storage that is still
  // allocated when the program exits, since the storage will look like a
  // memory leak.  So we have openDeviceCount as well as devicesOpen, and when
  // openDeviceCount becomes zero we set devicesOpen to zero after freeing
  // the memory.
  bool* devicesOpen = 0;
};

struct LinuxJoystick::LinuxJoystickData {
  int m_fd;
  int m_pos;
  // Whether we have observed an event that we have not reported.
  // Here's the scenario that leads to an observed-but-not-reported event:
  // Suppose the user is wiggling the joystick, and then presses a button.  We
  // want to gather together all of the wiggles and report them as one event
  // from readEvent.  At the same time we'll put the button press into the
  // m_pendingEvent so we can report that from the next call to readEvent.
  // Also, if we're getting a bunch of axis motion events, and we're still
  // gathering events, we set m_eventIsPending to indicate that we have
  // received motion events that we have to report before finishing.  In this
  // case m_pendingEvent is a motion event.
  bool m_eventIsPending;
  // The event we have observed but not reported, if any.
  struct js_event m_pendingEvent;
  // The axes.
  Dynavec <Float> m_axes;
  SP<JoystickEvent> m_writableEvent;
  LinuxJoystickData ()
    : m_eventIsPending (false)
  {}
  void stuffEvent (const struct js_event &jse);
};

LinuxJoystick::LinuxJoystick (int fd, int pos)
  : Joystick (getAxes (fd), getButtons (fd)),
    m_data (NEW (LinuxJoystickData ()))
{
  m_data->m_fd = fd;
  m_data->m_pos = pos;
  m_data->m_axes.extendTo (getAxes (fd), 0);
  // ref/deref just in case getVariableEvent does smart pointer operations on
  // this.
  ref();
  m_data->m_writableEvent =
    dynamic_cast <JoystickEvent *> (&*getVariableEvent ());
  deref ();
  assert (m_data->m_writableEvent);
}

LinuxJoystick::~LinuxJoystick () {
  if (0 != close (m_data->m_fd)) {
    perror ("close for Linux joystick");
    exit (1);
  }
  assert (devicesOpen);
  assert (devicesOpen [m_data->m_pos]);
  devicesOpen [m_data->m_pos] = false;
  delete m_data;
  m_data = 0;
  openDeviceCount --;
  if (0 == openDeviceCount) {
    delete devicesOpen;
    devicesOpen = 0;
  }
}

String LinuxJoystick::getName () {
  char name [128];
  // The man entry for ioctl says it returns 0 on success, but this one
  // apparently can return positive numbers on success.  Reading the driver
  // source code, it looks like it returns the length of the name
  if (0 > ioctl (m_data->m_fd, JSIOCGNAME (sizeof (name)), name)) {
    perror ("getting name of Linux joystick");
    exit (1);
  }
  return name;
}

Dynavec <String> LinuxJoystick::joystickDeviceNames () {
  return Dynavec <String> ("/dev/js0", "/dev/js1", "/dev/js2", "/dev/js3");
}

SP<LinuxJoystick> LinuxJoystick::openJoystickDevice (const String &deviceName)
{
  int pos;
  if (!VecUtil::find (deviceName, joystickDeviceNames(), &pos)) {
    message ("Device name " + deviceName+
	     " is not on the list of joystick device names");
    return 0;
  }
  if (devicesOpen && devicesOpen [pos]) {
    // It is not appropriate to have a warning here, since it happens in the
    // normal case when the SpaceOrb takes precedence over the LinuxJoystick.
    return 0;
  }
  int fd = open (&*deviceName, O_RDONLY | O_NDELAY);
  if (fd < 0) {
    if (ENODEV == errno || ENOENT == errno) {
      // No message, since this case is normal.
    } else {
      perror (&*("Opening " + String (deviceName)));
    }
    return 0;
  } 
  if (0 == openDeviceCount) {
    assert (0 == devicesOpen);
    const int size = joystickDeviceNames().size();
    devicesOpen = NEW (bool [size]);
    for (int i = 0; i < size; i ++) devicesOpen [i] = false;
    openDeviceCount++;
  }
  assert (!devicesOpen [pos]);
  devicesOpen [pos] = true;
  return NEW (LinuxJoystick (fd, pos));
}

bool LinuxJoystick::readEvent () {
  if (m_data->m_eventIsPending) {
    m_data->stuffEvent (m_data->m_pendingEvent);
    m_data->m_eventIsPending = false;
    return true;
  } else {
    for (;;) {
      // Loop to compress multiple motion events into one.
      struct js_event js;
      int bytes = read (m_data->m_fd, &js, sizeof (struct js_event));
      if (bytes < 0) {
	if (EAGAIN == errno) {
	  // No more input from the joystick.
	  if (m_data->m_eventIsPending) {
	    m_data->stuffEvent (m_data->m_pendingEvent);
	    m_data->m_eventIsPending = false;
	    return true;
	  } else {
	    return false;
	  }
	} else {
	  perror (&*("Error reading from joystick "+
		     joystickDeviceNames()[m_data->m_pos]));
	  // Should we return on this sort of error?  Exit the program?  Keep
	  // looping?  I don't know.
	  return false;
	}
      } else if (bytes != sizeof (struct js_event)) {
	message ("Ignoring wrong number of bytes read from joystick: "
		 +bytes);
	// Should we return on this sort of error?  Exit the program?  Keep
	// looping?  I don't know.
	return false;
      } else {
	// Drop the initial state events.  The initial state events are
	// generally a bunch of buttons going up, so by ignoring them we avoid
	// a bunch of "the button went up when it was not down" warnings,
	// thus making that warning message more useful.
	if (js.type & JS_EVENT_INIT) {
	  // Do nothing.  Our next action will be to read the next event, which
	  // is appropriate.
	} else if (JS_EVENT_AXIS == js.type) {
	  m_data->m_eventIsPending = true;
	  m_data->m_pendingEvent = js;
	  m_data->m_axes [js.number] = ((Float) js.value / (Float) 32767);
	} else if (JS_EVENT_BUTTON == js.type) {
	  if (m_data->m_eventIsPending) {
	    // There was pointer motion followed by button motion.  Report the
	    // pointer motion, and save the button motion in the pending event
	    // for next time.
	    assert (JS_EVENT_AXIS == m_data->m_pendingEvent.type);
	    m_data->stuffEvent (m_data->m_pendingEvent);
	    m_data->m_pendingEvent = js;
	    // m_eventIsPending should remain true.
	    return true;
	  } else {
	    // Button motion not preceeded by pointer motion.  Report the
	    // button motion and return.  m_eventIsPending should remain false.
	    m_data->stuffEvent (js);
	    return true;
	  }
	} else {
	  message ("Ignoring bad type from joystick: " +
		   (unsigned int) js.type);
	  // It seems harmless to keep looping around in this case.
	}
      }
    }
  }
}

void LinuxJoystick::LinuxJoystickData::stuffEvent (const struct js_event &jse)
{
  if (JS_EVENT_BUTTON == jse.type) {
    m_writableEvent->setButton (jse.number, 0 != jse.value);
  } else {
    if (JS_EVENT_AXIS != jse.type) {
      die (String ("Bad event type ") + jse.type +
	   " in LinuxJoystickData::stuffEvent");
    }
    m_writableEvent->setButton (m_writableEvent->pointerMotionButton (), true);
    m_writableEvent->setAxes (m_axes);
  }
}

// #define POLLING_TEST
#ifdef POLLING_TEST
#ifndef __iostream_h__
#include <iostream.h>
#define __iostream_h__
#endif
#endif

int LinuxJoystick::fdForSelect () const {
#ifdef POLLING_TEST
  static bool griped = false;
  if (!griped) {
    cout << "Remember to restore fdForSelect after testing" << endl;
    griped = true;
  }
  return -1; // Test polling.  Seems to work fine.
#else
  return m_data->m_fd;
#endif
}
