// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __KeyboardMouseEvent_h__
#include "KeyboardMouseEvent.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __Event_h__
#include "Event.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

namespace {
  typedef Event::Button Button;
  const Button FIRST = 0x80;
};
const Button KeyboardMouseEvent::MOTION = FIRST;
const Button KeyboardMouseEvent::LEFT_MOUSE = KeyboardMouseEvent::MOTION + 1;
const Button KeyboardMouseEvent::MIDDLE_MOUSE = KeyboardMouseEvent::LEFT_MOUSE + 1;
const Button KeyboardMouseEvent::RIGHT_MOUSE = KeyboardMouseEvent::MIDDLE_MOUSE + 1;
const Button KeyboardMouseEvent::WHEEL_UP = KeyboardMouseEvent::RIGHT_MOUSE + 1;
const Button KeyboardMouseEvent::WHEEL_DOWN = KeyboardMouseEvent::WHEEL_UP + 1;
const Button KeyboardMouseEvent::SHIFT = KeyboardMouseEvent::WHEEL_DOWN + 1;
const Button KeyboardMouseEvent::CONTROL = KeyboardMouseEvent::SHIFT + 1;
const Button KeyboardMouseEvent::META = KeyboardMouseEvent::CONTROL + 1;
const Button KeyboardMouseEvent::HOME = KeyboardMouseEvent::META + 1;
const Button KeyboardMouseEvent::DELETE = KeyboardMouseEvent::HOME + 1;
const Button KeyboardMouseEvent::END = KeyboardMouseEvent::DELETE + 1;
const Button KeyboardMouseEvent::BACKSPACE = KeyboardMouseEvent::END + 1;
const Button KeyboardMouseEvent::PAGEUP = KeyboardMouseEvent::BACKSPACE + 1;
const Button KeyboardMouseEvent::PAGEDOWN = KeyboardMouseEvent::PAGEUP + 1;
const Button KeyboardMouseEvent::RETURN = KeyboardMouseEvent::PAGEDOWN + 1;
const Button KeyboardMouseEvent::SPACE = KeyboardMouseEvent::RETURN + 1;
const Button KeyboardMouseEvent::TAB = KeyboardMouseEvent::SPACE + 1;
const Button KeyboardMouseEvent::ESCAPE = KeyboardMouseEvent::TAB + 1;
const Button KeyboardMouseEvent::LEFTARROW = KeyboardMouseEvent::ESCAPE + 1;
const Button KeyboardMouseEvent::RIGHTARROW = KeyboardMouseEvent::LEFTARROW + 1;
const Button KeyboardMouseEvent::DOWNARROW = KeyboardMouseEvent::RIGHTARROW + 1;
const Button KeyboardMouseEvent::UPARROW = KeyboardMouseEvent::DOWNARROW + 1;
namespace {
  const Button LAST = KeyboardMouseEvent::UPARROW + 1;
};

Button KeyboardMouseEvent::pointerMotionButton () const {
  return MOTION;
}

Button KeyboardMouseEvent::minButton () const {
  return FIRST;
}

Button KeyboardMouseEvent::maxButton () const {
  return LAST;
}

struct KeyboardMouseEvent::KeyboardMouseEventData {
  int m_x;
  int m_y;
  KeyboardMouseEventData () 
    : m_x (0), // Determinism is good.
      m_y (0)
  {
    // If we develop more than 128 buttons, the next assert might fire.
    assert (LAST > FIRST);
  }
};

KeyboardMouseEvent::KeyboardMouseEvent ()
  : Event (LAST),
    m_data (NEW (KeyboardMouseEventData ()))
{}

KeyboardMouseEvent::~KeyboardMouseEvent () {
  assert (m_data);
  delete m_data;
  m_data = 0;
}

int KeyboardMouseEvent::getX () const {
  return m_data->m_x;
}

void KeyboardMouseEvent::setX (int x) {
  m_data->m_x = x;
}

int KeyboardMouseEvent::getY () const {
  return m_data->m_y;
}

void KeyboardMouseEvent::setY (int y) {
  m_data->m_y = y;
}

namespace {
  bool staticIsValidButton (Button ch) {
    return (ch >= '!' && ch <= '~'
	    // Upper case keys are really lower case keys with the shift key
	    // down.
	    && (! (ch >= 'A' && ch <= 'Z')))
      || (ch >= FIRST && ch < LAST);
  }
};

bool KeyboardMouseEvent::isValidButton (Button ch) const {
  return staticIsValidButton (ch);
}

Button KeyboardMouseEvent::toButtonEvent (char ch) {
  switch (ch) {
  case '\177': return DELETE;
  case '\010': return BACKSPACE;
  case '\n':   return RETURN;
  case ' ':    return SPACE;
  case '\t':   return TAB;
  default:
    if (staticIsValidButton (ch)) {
      return (unsigned char) ch;
    } else {
      assert (0 && "Can't convert ch to a button");
      return MOTION;
    }
  }
}

String KeyboardMouseEvent::unParseButton (Button be) const {
  switch (be) {
  case MOTION: return "Mouse motion"; 
  case LEFT_MOUSE: return "Left mouse button"; 
  case MIDDLE_MOUSE: return "Middle mouse button"; 
  case RIGHT_MOUSE: return "Right mouse button";
    // IntelliMouse is a trademark of Microsoft, by the way.
  case WHEEL_UP: return "IntelliMouse wheel up"; 
  case WHEEL_DOWN: return "IntelliMouse wheel down"; 
  case SHIFT: return "Shift"; 
  case CONTROL: return "Control"; 
  case META: return "Meta or Alt"; 
  case HOME: return "Home"; 
  case DELETE: return "Delete"; 
  case END: return "End"; 
  case BACKSPACE: return "Back Space"; 
  case PAGEUP: return "Page Up"; 
  case PAGEDOWN: return "Page Down"; 
  case RETURN: return "Return or Enter"; 
  case SPACE: return "Space"; 
  case TAB: return "Tab";
  case ESCAPE: return "Escape";
  case LEFTARROW: return "Left Arrow"; 
  case RIGHTARROW: return "Right Arrow"; 
  case DOWNARROW: return "Down Arrow"; 
  case UPARROW: return "Up Arrow"; 
  case LAST:
    assert (0 && "There is no good reason to get the name of LAST");
    return "Last Button, returned under duress";
  default:
    assert (isValidButton (be));
    return String ((char) be);
  }
}

String KeyboardMouseEvent::unParse () const {
  if (getButton () != pointerMotionButton()) {
    return unParseButton (getButton()) + " " + (isDown()?"down":"up");
  } else {
    return String ("Mouse moved to ") + getX() + ", " + getY();
  }
}

