// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __JoystickEvent_h__
#define __JoystickEvent_h__

#ifndef __Event_h__
#include "Event.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

class String;

// A generic joystick event.  If and when we do a Windows port, we should have
// exactly the same joystick events whether it's under Windows or Linux.
// The comments here specify how the axes and buttons are laid out.  The hope
// is that if all available input drivers are consistent with the comments
// below, then a command that works when a particular kind of joystick is
// plugged in under Linux will work exactly the same with the same kind of
// joystick under Windows.
//
// The most likely way for this to fail would be that the buttons or axes are
// in a different order on the two systems.
class JoystickEvent
  : public Event
{
  struct JoystickEventData;
  JoystickEventData *m_data;
public:
  JoystickEvent (int axes, int buttons);
  ~JoystickEvent ();
  // maxButton will return buttons + 1.  The real buttons are numbered 0
  // through buttons - 1, and when we're doing pointer motion, that's button
  // number buttons.
  Button maxButton () const;
  // minButton will return 0.
  Button minButton () const;
  // pointerMotionButton will return buttons.
  Button pointerMotionButton () const;
  // isValidButton just checks that it's in the range 0 to maxButton() - 1.
  bool isValidButton (Button ch) const;
  // Call this to get a copy of the present value of the axes.  We take a
  // dynavec reference as input and update it, instead of returning a dynavec,
  // so code can use a JoystickEvent without chronically doing memory
  // allocation.  
  // Axes all have the range -1.0 to 1.0.  To find out how many axes, look at
  // the size of the dynavec.  Initially, the value for all of the axes is 0.
  void getAxes (Dynavec <Float> &axes) const;
  // This one should be used by an InputDevice to set the axes.  Debug code
  // will assert that the incoming vector has the number of axes that was given
  // to the constructor.
  void setAxes (const Dynavec <Float> &axes);
  // Returns "Button 0" through "Button n", for n = maxButton () - 1, or
  // "Joystick motion" if b is the pointerMotionButton.
  String unParseButton (Button b) const;
  String unParse () const;
};
#endif
