// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __JoystickEvent_h__
#include "JoystickEvent.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

typedef Event::Button Button;

struct JoystickEvent::JoystickEventData {
  int m_buttons;
  Dynavec <Float> m_axes;
};

JoystickEvent::JoystickEvent (int axes, int buttons)
  : Event (buttons + 1),
    m_data (NEW (JoystickEventData ()))
{
  assert (axes >= 0);
  assert (buttons >= 0);
  m_data->m_axes.extendTo (axes, 0.0);
  m_data->m_buttons = buttons;
}

JoystickEvent::~JoystickEvent () {
  assert (m_data);
  delete m_data;
  m_data = 0;
}

Button JoystickEvent::maxButton () const {
  return m_data->m_buttons + 1;
}

Button JoystickEvent::minButton () const {
  return 0;
}

Button JoystickEvent::pointerMotionButton () const {
  return m_data->m_buttons;
}

bool JoystickEvent::isValidButton (Button ch) const {
  // ch is an unsigned char, so it is always the case that 0 <= ch.
  return ch < maxButton ();
}

void JoystickEvent::getAxes (Dynavec <Float> &axes) const {
  axes = m_data->m_axes;
}

void JoystickEvent::setAxes (const Dynavec <Float> &axes) {
  assert (axes.size() == m_data->m_axes.size());
  m_data->m_axes = axes;
}

String JoystickEvent::unParseButton (Button b) const {
  assert (isValidButton (b));
  if (b == pointerMotionButton ()) {
    return "Joystick motion";
  } else {
    return "Button " + String ((int) b);
  }
}

String JoystickEvent::unParse () const {
  if (getButton () != pointerMotionButton ()) {
    return unParseButton (getButton()) + " " + (isDown()?"down":"up");
  } else {
    String result = "Joystick moved to ";
    for (int i = 0; i < m_data->m_axes.size(); i++) {
      result += String (m_data->m_axes [i]);
      if (i < m_data->m_axes.size()-1) {
	result += ", ";
      }
    }
    return result;
  }
}
