// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __InputDevice_h__
#define __InputDevice_h__

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __Configurable_h__
#include "Configurable.h"
#endif

#ifndef __TopLevel_h__
#include "TopLevel.h"
#endif

class Event;
class RecursiveSlotValue;
class InputDeviceConfiguration;

class InputDevice
  : public Configurable
{
  struct InputDeviceData;
  InputDeviceData *m_data;
protected:
  InputDevice (InputDeviceConfiguration *conf, Event *event);
  // Subclasses are responsible for doing the input, so we need to let them
  // change the event.  That's the same event they gave us, but I couldn't
  // figure out a reentrant way for them to retain the pointer to the event
  // they gave us, so let's just plan to give it back to them when they want
  // it.
  SP<Event> getVariableEvent ();
public:
  ~InputDevice ();
  // returns a pointer to the current event from the InputDevice.  Doesn't do
  // any I/O.  The event is overwritten in place each time new input happens on
  // this device.
  SP<const Event> getEvent () const;
  // Return the configuration that was passed to the constructor.
  SP<InputDeviceConfiguration> getConf ();
  SP<const InputDeviceConfiguration> getConf () const;
  // Grabs the input device.  See the comments in TopLevel::grab, which will
  // simply dispatch to this method based on which device gave the last input.
  void grab (SP<RecursiveSlotValue> rsv);
  // Dispatches based on the current event.
  void dispatchEvent (TopLevel *top);
  // If there's an event, then readEvent reads it, returns true, and copies the
  // event it read into the InputDevice so you can get to it with getEvent.  If
  // there is no event, then readEvent returns false.  readEvent never waits
  // for anything.
  virtual bool readEvent () = 0;
  // fdForSelect returns the file descriptor to use when calling select to find
  // the next input device that has input available, or -1 if the select won't
  // work for the device and the device needs to be polled.
  //
  // Sometimes I though this should go in a separate class "SelectInputDevice"
  // which had InputDevice as a virtual base class.  I flipped back and forth
  // between having it there and having it here.  The advantages of the
  // separate class were:
  //
  // 1. It caused input devices that didn't have a meaningful file descriptor
  //    to not have the useless method in their interface.
  // 2. It was a good use for virtual inheritance.
  // 
  // and the reasons it ultimately wound up here were:
  //
  // 1. g++ bugs that kept me from doing virtual inheritance the way I wanted.
  // 2. I wanted to be able to have a class that wrapped around an InputDevice
  //    and presented a spruced-up InputDevice, specifically a SpaceOrb as a
  //    spruced-up joystick.  If there were multiple classes of InputDevices,
  //    then I would need multiple classes of SpaceOrb's.  This last one is the
  //    one that made me sure I wanted fdForSelect here.
  virtual int fdForSelect () const = 0;
};

#endif
