// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __InputDevice_h__
#include "InputDevice.h"
#endif

#ifndef __InputDeviceConfiguration_h__
#include "InputDeviceConfiguration.h"
#endif

#ifndef __Event_h__
#include "Event.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __RecursiveSlotValue_h__
#include "RecursiveSlotValue.h"
#endif

#ifndef __TopLevel_h__
#include "TopLevel.h"
#endif

#ifndef __ActionConfiguration_h__
#include "ActionConfiguration.h"
#endif

#ifndef __iostream_h__
#include <iostream.h>
#define __iostream_h__
#endif

#ifndef __ConstantRecursiveSlotValue_h__
#include "ConstantRecursiveSlotValue.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __Action_h__
#include "Action.h"
#endif

#ifndef __DispatcherConfiguration_h__
#include "DispatcherConfiguration.h"
#endif

typedef Event::Button Button;

struct InputDevice::InputDeviceData {
  SP<InputDeviceConfiguration> m_conf;
  SP<Event> m_event;
  SP<RecursiveSlotValue> m_grab;
  // The button they pressed at the beginning of the grab, so we know when to
  // un-grab.
  Button m_startButton;
  void dispatchEvent (TopLevel *top);
};

InputDevice::InputDevice (InputDeviceConfiguration *conf, Event *event)
  : m_data (NEW (InputDeviceData ()))
{
  m_data->m_conf = conf;
  m_data->m_event = event;
  void dispatchEvent (TopLevel *top);
}

InputDevice::~InputDevice () {
  assert (m_data);
  delete m_data;
  m_data = 0;
}

SP<const Event> InputDevice::getEvent () const {
  return &*m_data->m_event;
}

SP<Event> InputDevice::getVariableEvent () {
  return &*m_data->m_event;
}

void InputDevice::grab (SP<RecursiveSlotValue> rsv) {
  m_data->m_grab = rsv;
}

void InputDevice::dispatchEvent (TopLevel *top) {
  m_data->dispatchEvent (top);
}

void InputDevice::InputDeviceData::dispatchEvent (TopLevel *top) {
  SP<RecursiveSlotValue> oldGrab = m_grab;
  m_grab = 0;
  SP<Configurable> action = 0;
  if (oldGrab) {
    // Dispatch to the grabbing action.
    SP<ActionConfiguration> ac =
      dynamic_cast <ActionConfiguration *>
      (&*oldGrab->getConfiguration());
    assert (ac);
    ac->setTopLevel (top);
    action = oldGrab->makeIt();
    if (!action) {
      cout << "Running the command produced a null action." << endl;
      // Should we ungrab in this case?  It's sort of an error situation,
      // so yes.
      oldGrab = 0;
      m_grab = 0;
    }
  } else {
    // Dispatch based on the event.
    SP<RecursiveSlotValue> crsv = m_conf->getDispatcher();
    SP<ActionConfiguration> dispatcherConf =
      dynamic_cast <ActionConfiguration *> (&*crsv->getConfiguration());
    assert (dispatcherConf);
    dispatcherConf->setTopLevel (top);
    action = crsv->makeIt ();
    // Beware -- action can be null if the key was not bound.
    if (!action &&
	// Don't generate warnings on unbound pointer motion.
	m_event->getButton () != m_event->pointerMotionButton () &&
	dynamic_cast <DispatcherConfiguration *> (&*dispatcherConf)) {
      // FIXME Message should mention which shift keys are down
      cout <<"The button "<<m_event->unParseButton (m_event->getButton())
	   <<" has no meaning here."<<endl
	   << "The meaningful buttons are:" << endl
	   << *dynamic_cast <DispatcherConfiguration *> (&*dispatcherConf)
	// Printing dispatcherConf should end in a newline.  
	// Add another newline here, so if they bang on a bad key
	// the repeated error messages don't blur into a mush.
	   << endl;
    }
  }
  if (m_grab && !oldGrab) {
    // if it's a new grab, then decide when to automatically ungrab
    if (m_event->isDown ()) {
      m_startButton = m_event->getButton ();
    } else {
#ifndef NDEBUG
      cout << "Starting grab without a clear idea "
	   << "about when to end the grab." << endl;
      cout << "Will end the grab when no buttons are down." << endl;
#endif
      m_startButton = m_event->maxButton ();
    }
  }
  if (oldGrab && !m_grab) {
    // They had grabbed focus before, but they didn't explictly regrab.
    // The automatic grab is over when either the button that started
    // the grab is released, or if we're confused then at least it's
    // over when all of the buttons are released.
    // 
    // I used to think that the grab should be over when all buttons are
    // released, but this is wrong.  Consider an action that wants mouse motion
    // events and is bound to shift left mouse.  Then if you hold down the
    // shift button, then press the left mouse button, then wiggle the mouse,
    // then release the left mouse button, then wiggle the mouse, then release
    // the shift button, you want the first batch of wiggles to have an effect
    // but not the second batch.
    if (!((!m_event->isDown() && m_event->getButton() == m_startButton)
	  || 0 == m_event->numButtonsDown ())) {
      m_grab = oldGrab;
    }
  }
  if (action) {
    SP<Action> trueAction = dynamic_cast <Action *> (&*action);
    assert (trueAction);
    if (trueAction->isProblem()) {
      cout << trueAction->getProblem() << endl;
      m_grab = 0;
    } else if (trueAction->isMessage ()) {
      cout << trueAction->getMessage() << endl;
    }
  }
}

SP<const InputDeviceConfiguration> InputDevice::getConf () const {
  return &*m_data->m_conf;
}

SP<InputDeviceConfiguration> InputDevice::getConf () {
  return m_data->m_conf;
}

