// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __Event_h__
#define __Event_h__

#ifndef __Refcount_h__
#include "Refcount.h"
#endif

class String;
class ostream;

class Event
  : public Refcount
{
  // This class represents actions taken by the user.  The intent is that one
  // Event instance will be modified in place to report many events.
  struct EventData;
  EventData *m_data;
public:
  // Printable characters are themselves when interpreted as buttons.
  // Shift keys are ignored; so a capital A will be the button 'a', pressed
  // when BUTTON_SHIFT is down.
  typedef unsigned char Button;
  Event (Button maxButton);
  ~Event ();

  // Many input devices have pointers, and we need a button to report when the
  // pointer moves.  This button is different from the others because it
  // doesn't go up and down, it's just a place holder.  We want the pointer
  // motion button to be a button so we can grab it just like all of the
  // others.  The pointerMotionButton goes down, but it never goes up.
  virtual Button pointerMotionButton () const = 0;
  virtual Button maxButton () const = 0;
  virtual Button minButton () const = 0;

  Button getButton () const;
  void setButton (Button ch, bool down);
  // Buttons and keys can go up and down.  If the down flag is true, then this
  // event is reporting that the button went down.  Don't confuse this
  // with methods elsewhere that report whether the button or key *is*
  // presently down due to some event in the past.  Some buttons and keys
  // autorepeat; in this case it may go down many times, and eventually it
  // should go up when the user lets go of it.
  bool isDown () const;

  // Whether the Button is one that we will ever get.  For instance,
  // isValidButton (' ') will return false, since the button is BUTTON_SPACE.
  // We want this method so we can discourage the user from binding actions to
  // buttons that can never happen.

  virtual bool isValidButton (Button ch) const = 0;
  // Return the name of a button, so we can have a dialogue with the user about
  // customizing buttons.  The client code shouldn't call this with an invalid
  // button; debug code should abort in that case.
  virtual String unParseButton (Button b) const = 0;
  // But we can provide a useful default implementation for converting strings
  // to buttons.  It just loops over all of the known buttons, converts them to
  // strings, and looks for a match.  Returns maxButton() if the string could
  // not be parsed. 
  virtual Button parseButton (const String &buttonName) const;

  // Generally we call setKeyDown and setButtonDown and then dispatch the
  // event, so by the time we get into the Action that we dispatched to, the
  // downness of the button or key we dispatched has already been
  // incorporated into the result from isKeyDown and isButtonDown.  This way,
  // an Action can know that it's about to lose it's grab on the user events,
  // because it can observe that numKeysAndButtonsDown has decreased to zero.
  // Beware that for buttons like shift, where there are 
  // more than one, it may not be the case that immediately
  // after releasing the button, the button is up, since it's possible for the
  // other instance of the button to still be down.

  // Note that X does it the other way around, incorporating events into the
  // mask after giving the events to the user.  So we have to do a little extra
  // work to do it this way.  I think we really do want it this way.

  // For a newly created event, no buttons or keys are down.

  // Don't call the next two with BUTTON_MOTION or BUTTON_LAST.
  // Only call the next two with valid characters, per isValidChar.

  bool isButtonDown (Button be) const;
  void setButtonDown (Button be, bool isDown);
  // The number of keys and buttons down.  When this goes down to zero, the
  // TopLevel knows to remove the grab if it didn't reassert its grab, and
  // the Action knows that it had better regrab if it wants to retain control.
  int numButtonsDown () const;
  // Convert the event to a string, for debugging.
  virtual String unParse () const = 0;
  // Calls unParse and prints the resulting string.
  friend ostream &operator<< (ostream &o, const Event& e);
};

#endif
