// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __Event_h__
#include "Event.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __iostream_h__
#include <iostream.h>
#define __iostream_h__
#endif

typedef Event::Button Button;

struct Event::EventData {
  bool m_down;
  Button m_button;
  Dynavec <bool> m_downButtons;
  // Number of keys and buttons that are down.
  int m_downCount;
  EventData ()
    // Initializing only because determinism is good.  These values aren't
    // particularly useful.
    : m_down (false),
      m_button (0),
      m_downCount (0)
  {}
  inline void downCheck () {
#ifndef NDEBUG
    int total = 0;
    for (int i = 0; i < m_downButtons.size(); i++) {
      if (m_downButtons [i]) total++;
    }
    assert (total == m_downCount);
#endif
  }
};

Event::Event (Button maxButton)
  : m_data (NEW (EventData ()))
{
  // Just in case maxButton uses smart pointers to this.
  ref ();
  m_data->m_downButtons.extendTo (maxButton, false);
  deref ();
}

Event::~Event ()
{
  assert (m_data);
  delete m_data;
  m_data = 0;
}

Button Event::getButton () const {
  assert (isValidButton (m_data->m_button));
  return m_data->m_button;
}

void Event::setButton (Button be, bool down) {
  assert (isValidButton (be));
  // Pointer motion goes down, it never goes up.
  assert ((pointerMotionButton () != be) || down);
  m_data->m_button = be;
  if (pointerMotionButton () != be &&
      !down &&
      !isButtonDown (m_data->m_button)) {
    cerr << "Warning -- the button "
	 << unParseButton (m_data->m_button)
	 << " went up but it is not down."
	 << endl;
  }
  m_data->m_down = down;
  if (be != pointerMotionButton ()) {
    setButtonDown (be, down);
  }
}

bool Event::isDown () const {
  return m_data->m_down;
}

// Eliminate information from a string that we don't want to bother us when
// we're parsing button names from the user.  Specifically, convert to lower
// case, and eliminate spaces and tabs.
static String normalize (const String &s) {
  String result;
  for (int i = 0; i < s.size(); i++) {
    if ('A' <= s[i] && 'Z' >= s[i]) {
      result += s[i] - 'A' + 'a';
    } else if (' ' == s[i] || '\t' == s[i]) {
      // Do nothing;
    } else {
      result += s[i];
    }
  }
  return result;
}

// Parse by comparing with all possibilities.  This doesn't have to run very
// fast.
Button Event::parseButton (const String &s) const {
  if (1 == s.size() && isValidButton ((Button) s[0])) {
    return (Button) s[0];
  } else {
    const String norms = normalize (s);
    for (Button b = minButton (); b < maxButton (); b++) {
      if (isValidButton (b) && normalize (unParseButton (b)) == norms) {
	return b;
      }
    }
    return maxButton ();
  }
}

bool Event::isButtonDown (Button be) const {
  assert (be != pointerMotionButton ());
  assert (isValidButton (be));
  return m_data->m_downButtons [be];
}

void Event::setButtonDown (Button be, bool isDown) {
  assert (be != pointerMotionButton ());
  assert (isValidButton (be));
  bool &buttonDown = m_data->m_downButtons [be];
  if (isDown) {
    if (!buttonDown) {
      m_data->m_downCount ++;
    }
  } else {
    if (buttonDown) {
      m_data->m_downCount --;
    }
  }    
  buttonDown = isDown;
  m_data->downCheck ();
}

int Event::numButtonsDown () const {
  m_data->downCheck ();
  return m_data->m_downCount;
}

ostream &operator<< (ostream &o, const Event& e) {
  return o << e.unParse ();
}
