// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#include "DispatcherConfiguration.h"

#ifndef __Configuration_h__
#include "Configuration.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __StringSlotValue_h__
#include "StringSlotValue.h"
#endif

#ifndef __RecursiveSlotValue_h__
#include "RecursiveSlotValue.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __VectorSlotValue_h__
#include "VectorSlotValue.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __Configurable_h__
#include "Configurable.h"
#endif

#ifndef __ConstantRecursiveSlotValue_h__
#include "ConstantRecursiveSlotValue.h"
#endif

#ifndef __Event_h__
#include "Event.h"
#endif

#ifndef __ButtonSlotValue_h__
#include "ButtonSlotValue.h"
#endif

#ifndef __VecUtil_h__
#include "VecUtil.h"
#endif

namespace {
  typedef Event::Button Button;
  class KeyBinding
    : public Configuration
  {
    static String keyName () {
      return "Key";
    }

    static String factoryName () {
      return "What it should do";
    }
    String m_protocol;
  public:
    // Unfortunately, the factory protocol requires us to be able to make a
    // default configuration without knowing anything, so we can't pass the
    // event into the KeyBinding constructor.  Thus we have to separate the
    // constructor from the initializeSlots method.
    KeyBinding (const String &protocol)
      : m_protocol (protocol)
    {}
    void initializeSlots (SP<const Event> event) {
      initializeSlot (keyName(), NEW (ButtonSlotValue (event)));
      initializeSlot (factoryName(),
		      NEW (RecursiveSlotValue (m_protocol,
					       "DoNothing")));
    }    
    KeyBinding (const KeyBinding &kb)
      : Configuration (kb)
    {}
    Button getKey () const {
      return ButtonSlotValue::getSlot (this, keyName ());
    }
    void setKey (Button ch) {
      ButtonSlotValue::setSlot (this, keyName (), ch);
    }
    SP<RecursiveSlotValue> getBinding () {
      return RecursiveSlotValue::getSlotValue (this, factoryName());
    }
    SP<const RecursiveSlotValue> getBinding () const {
      return RecursiveSlotValue::getSlotValue (this, factoryName());
    }
    void setBinding (SP<RecursiveSlotValue> rsv) {
      RecursiveSlotValue::setSlotValue (this, factoryName(), rsv);
    }
    bool hasTheSameTypeAs (const Configuration *c) const {
      return 0 != dynamic_cast <const KeyBinding *> (c);
    }
    SP<Configuration> copy () const {
      return NEW (KeyBinding (*this));
    }
    String unParse () const {
      SP<const SlotValue> slot = getSlot (keyName ());
      assert (slot);
      String keyName = slot->unParse ();
      // I think the next three lines could have been replaced by the following
      // one, but g++ doesn't like it, and I couldn't quickly figure out why.
      // String keyName = getSlot (keyName ())->unParse ();
      return keyName + " is bound to " + getBinding()->unParse();
    }
  };

  // Return the binding for ch if it is found, otherwise null.
  SP<KeyBinding> getKeyBinding (SP<VectorSlotValue> bindings, Button key)
  {
    for (int i = 0; i < bindings->size(); i++) {
      SP<SlotValue> sv = (*bindings)[i];
      SP<ConstantRecursiveSlotValue> crsv =
	dynamic_cast <ConstantRecursiveSlotValue *> (&*sv);
      SP<KeyBinding> kb =
	dynamic_cast <KeyBinding *> (&*crsv->getConfiguration());
      assert (kb);
      if (key == kb->getKey()) {
	return kb;
      }
    }
    return 0;
  }

  class KeyBindingFactory
    : public Factory
  {
    String m_protocol;
  public:
    KeyBindingFactory (const String &protocol)
      : Factory ("KeyBinding"),
	m_protocol (protocol)
    {}
    SP<Configuration> defaultConfiguration () const {
      // The KeyBinding made here is less than satisfactory, since it doesn't
      // have its slots initialized yet.
      return NEW (KeyBinding (m_protocol));
    }
    SP<Configurable> makeIt (SP<Configuration> conf) const {
      (void) conf;
      // You never make a keybinding, you just configure it.
      // But we need to have a factory here because ConstantRecursiveSlotValue
      // expects one.  We want to use ConstantRecursiveSlotValue because the
      // configuration editor expects it to be there, and we definitely want to
      // use the configuration editor to edit key binding tables.
      //
      // FIXME ConstantRecursiveSlotValue needs subclass w/o factory slot.
      //
      // Call it ConfigurationSlotValue.  I created these silly unused
      // factories when I made lists of input devices in the input devices
      // factories too; clean up both places.
      abort ();
      // Compiler doesn't know that abort doesn't return.
      return 0;
    }
  };

  SP<ConstantRecursiveSlotValue> newKeyBinding (SP<const Event> event,
						const String &protocol) {
    SP<ConstantRecursiveSlotValue> result =
      NEW (ConstantRecursiveSlotValue
	   ("KeyBinding", NEW (KeyBindingFactory (protocol))));
    SP<Configuration> conf = result->getConfiguration();
    SP<KeyBinding> kbconf = dynamic_cast <KeyBinding *> (&*conf);
    kbconf->initializeSlots (event);
    return result;
  }
} // namespace

struct DispatcherConfiguration::DispatcherConfigurationData {
  // An event of the approriate class, which we can use to parse and unparse
  // buttons and so forth.
  SP<const Event> m_event;
  // The specific superclass of Action that we are willing to let the user put
  // into this dispatch table.  Typically "KeyboardMouseAction" or
  // "SpaceOrbAction".
  String m_protocol;
  // All of the shift keys.
  // FIXME Could allow the user to configure the set of shift keys.
  Dynavec <Event::Button> m_shiftButtons;
  String bindingSlotName (Button b) {
    if (m_event->maxButton() == b) {
      return "Bindings with no shift keys";
    } else {
      return
	("Bindings when " + m_event->unParseButton (b)
	 + " is down");
    }
  }
};

DispatcherConfiguration::DispatcherConfiguration
(SP<const Event> event,
 const String &protocol,
 const Dynavec <Button> &shiftButtons)
  : m_data (NEW (DispatcherConfigurationData ()))
{
  m_data->m_event = event;
  m_data->m_protocol = protocol;
  // Add maxButton () to the beginning of the list of shift keys, so the
  // un-shifted case isn't a special case.  Add it to the beginning so the
  // simple keys come out first when we list all keys.
  m_data->m_shiftButtons.push (event->maxButton ());
  for (int i = 0; i < shiftButtons.size(); i++) {
    m_data->m_shiftButtons.push (shiftButtons [i]);
  }
  for (int i = 0; i < m_data->m_shiftButtons.size(); i++) {
    initializeSlot (m_data->bindingSlotName (m_data->m_shiftButtons [i]),
		    NEW (VectorSlotValue (&*newKeyBinding (&*event,
							   protocol))));
  }
}

DispatcherConfiguration::~DispatcherConfiguration () {
  assert (m_data);
  delete m_data;
  m_data = 0;
}

DispatcherConfiguration::DispatcherConfiguration
(const DispatcherConfiguration &dc)
  : ActionConfiguration (dc),
    m_data (NEW (DispatcherConfigurationData (*dc.m_data)))
{}

SP<Configuration> DispatcherConfiguration::copy () const {
  return NEW (DispatcherConfiguration (*this));
}

bool DispatcherConfiguration::hasTheSameTypeAs (const Configuration *c)
  const
{
  return 0 != dynamic_cast <const DispatcherConfiguration *> (c);
}

void DispatcherConfiguration::bindButton (Button key, RecursiveSlotValue *a_f)
{
  bindButton (key, a_f, m_data->m_event->maxButton ());
}

void DispatcherConfiguration::bindButton (Button key, RecursiveSlotValue *a_f,
					  Button shift) {
  SP<RecursiveSlotValue> f = a_f;
  String name;
  if (!VecUtil::find (shift, m_data->m_shiftButtons)) {
    assert (0 && "Bad shift value in bindKey");
    // Ignore the bad binding.
  } else {
    name = m_data->bindingSlotName (shift);
    SP<VectorSlotValue> vsv =
      dynamic_cast <VectorSlotValue *> (&*getSlot (name));
    SP<KeyBinding> binding = getKeyBinding (vsv, key);
    if (binding) {
      binding->setBinding (f);
    } else {
      int where = vsv->size();
      vsv->extendTo (where + 1);
      SP<ConstantRecursiveSlotValue> newBinding =
	newKeyBinding (m_data->m_event,
		       m_data->m_protocol);
      SP<KeyBinding> kb =
	dynamic_cast <KeyBinding *> (&*newBinding->getConfiguration());
      assert (kb);
      kb->setKey (key);
      kb->setBinding (f);
      vsv->setValue (where, &*newBinding);
    }
  }
}

void DispatcherConfiguration::bindButton
(Button key, const String &factoryName, Button shift) {
  bindButton (key,
	      NEW (RecursiveSlotValue (m_data->m_protocol, factoryName)),
	      shift);
}

void DispatcherConfiguration::bindButton
(Button key, const String &factoryName) {
  bindButton (key, factoryName, m_data->m_event->maxButton ());
}

SP<RecursiveSlotValue> DispatcherConfiguration::getBinding
(SP<const Event> event)
{
  // Should only call us when the mouse moved or a key went up.
  // Calling this when a key goes up is weird.  
  assert (event->getButton() == m_data->m_event->pointerMotionButton () ||
	  event->isDown());
  const Button maxButton = m_data->m_event->maxButton ();
  assert (event->maxButton () == maxButton);
  bool found = false;
  // If we don't initialize whichShift here, the compiler thinks it may be read
  // before written.  This isn't true, but no use arguing.
  Button whichShift = 0;
  for (int i = 0; i < m_data->m_shiftButtons.size(); i++) {
    // If the shift button at position i is down (or it is maxButton, which
    // means we're taking the non-shifted interpretation), and all of the other
    // shift buttons are up, then we've discovered which binding map to use.
    const Button thisButton = m_data->m_shiftButtons [i];
    if ((thisButton == maxButton) || event->isButtonDown (thisButton)) {
      // Okay, this button is down, so if all of the other shift buttons are
      // up, we're done.  But be sure not to query whether maxButton is up, or
      // whether this button is up.
      bool otherButton = false;
      for (int j = 0; j < m_data->m_shiftButtons.size(); j++) {
	const Button thatButton = m_data->m_shiftButtons [j];
	if (thatButton != thisButton &&
	    thatButton != maxButton &&
	    event->isButtonDown (thatButton)) {
	  otherButton = true;
	}
      }
      if (!otherButton) {
	found = true;
	whichShift = thisButton;
      }
    }
  }
  if (! found) {
    return 0;
  } else {
    String name = m_data->bindingSlotName (whichShift);
    SP<VectorSlotValue> rsv =
      dynamic_cast <VectorSlotValue *> (&*getSlot (name));
    SP<KeyBinding> binding = getKeyBinding (rsv, event->getButton ());
    if (binding) {
      return binding->getBinding ();
    } else {
      return 0;
    }
  }
}

const Dynavec <Button> &DispatcherConfiguration::getShiftButtons () const {
  return m_data->m_shiftButtons;
}


ostream &operator<< (ostream &o,
		     const DispatcherConfiguration &dc)
{
  Dynavec <Dynavec <String> > tableau;
  tableau.push (Dynavec <String> ());
  tableau[0].push ("Key");
  tableau[0].push ("Action");
  for (int j = 0; j < dc.m_data->m_shiftButtons.size(); j++) {
    const Button shiftButton = dc.m_data->m_shiftButtons [j];
    String slotName = dc.m_data->bindingSlotName (shiftButton);
    String prefix;
    if (dc.m_data->m_event->maxButton () == shiftButton) {
      // If there's no shift key, we have an empty prefix.
      prefix = "";
    } else {
      // Otherwise, if (for example) the meta key is pushed, the prefix is
      // "Meta ".
      prefix = dc.m_data->m_event->unParseButton (shiftButton) + " ";
    }
    SP<const VectorSlotValue> rsv =
      dynamic_cast <const VectorSlotValue *> (&*dc.getSlot (slotName));
    for (int i = 0; i < rsv->size(); i++) {
      SP<const ConstantRecursiveSlotValue> crsv =
	dynamic_cast <const ConstantRecursiveSlotValue *> (&*((*rsv)[i]));
      assert (crsv);
      SP<const KeyBinding> kb =
	dynamic_cast <const KeyBinding *>  (&*crsv->getConfiguration());
      assert (kb);
      Button key = kb->getKey();
      tableau.push (Dynavec <String> ());
      // If the meta-a key is pushed, then prefix will be "Meta ", so we
      // will put "Meta a" on the left column of the table.
      tableau[tableau.size()-1].push (prefix +
				      dc.m_data->m_event->unParseButton (key));
      tableau[tableau.size()-1].push (kb->getBinding()->unParse());
    }
  }
  StreamUtil::printTableau (o, tableau);
  return o;
}

