// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __Dispatcher_h__
#include "Dispatcher.h"
#endif

#ifndef __DispatcherConfiguration_h__
#include "DispatcherConfiguration.h"
#endif

#ifndef __Factory_h__
#include "Factory.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __RecursiveSlotValue_h__
#include "RecursiveSlotValue.h"
#endif

#ifndef __Configurable_h__
#include "Configurable.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __ActionConfiguration_h__
#include "ActionConfiguration.h"
#endif

#ifndef __TopLevel_h__
#include "TopLevel.h"
#endif

#ifndef __ScaleConfiguration_h__
#include "ScaleConfiguration.h"
#endif

#ifndef __VecUtil_h__
#include "VecUtil.h"
#endif

#ifndef __Action_h__
#include "Action.h"
#endif

#ifndef __KeyboardMouseEvent_h__
#include "KeyboardMouseEvent.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __Event_h__
#include "Event.h"
#endif

struct Dispatcher::DispatcherData {
  SP<const Event> m_event;
  String m_protocol;
};

Dispatcher::Dispatcher (const String &factoryName,
			const String &protocolName,
			SP<const Event> event)
  : Factory (factoryName),
    m_data (NEW (DispatcherData ()))
{
  m_data->m_event = event;
  m_data->m_protocol = protocolName;
}

Dispatcher::~Dispatcher () {
  assert (m_data);
  delete m_data;
  m_data = 0;
}

SP<Configuration> Dispatcher::defaultConfiguration () const {
  return NEW (DispatcherConfiguration (m_data->m_event,
				       m_data->m_protocol,
				       Dynavec <Event::Button> ()));
}

SP<Configurable> Dispatcher::makeIt (const SP<Configuration> c)
  const
{
  SP <DispatcherConfiguration> conf =
    dynamic_cast <DispatcherConfiguration *> (&*c);
  assert (conf);
  SP<const Event> event = conf->getTopLevel()->getEvent ();
  if (event->getButton () != event->pointerMotionButton () &&
      !event->isDown ()) {
    // If the event is not pointer motion, and it's a button going up,
    // then declare success. 
    return NEW (Action ());
  } else if (VecUtil::find (event->getButton (),
			    conf->getShiftButtons ())) {
    // If the event is a shift button going down, declare success.
    return NEW (Action ());
  } else {
    SP<RecursiveSlotValue> binding =
      conf->getBinding (event);
    if (binding) {
      SP<ActionConfiguration> ac =
	dynamic_cast <ActionConfiguration *> (binding->getConfiguration ());
      assert (ac);
      ac->setTopLevel (conf->getTopLevel());
      SP<Configurable> result = binding->makeIt();
      return result;
    } else {
      // Return a null action when the key is not bound, so the
      // caller can easily recognize this circumstance and either
      // generate an error message or give some other interpretation
      // to the key.
      return 0;
    }
  }
}
