// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __ButtonSlotValue_h__
#define __ButtonSlotValue_h__

#ifndef __SlotValue_h__
#include "SlotValue.h"
#endif

#ifndef __Event_h__
#include "Event.h"
#endif

class String;
class Configuration;
class ButtonSlotValue
  : public SlotValue
{
  struct ButtonSlotValueData;
  ButtonSlotValueData *m_data;
public:
  // This takes an event as an argument so we can use it to parse and unparse
  // button names.  Unfortunately, since we really need this, we can't make a
  // Dynavec of ButtonSlotValue's, since Dynavec requires a default constructor
  // with no arguments.  Alternatives are:
  // 1. Have a flavor of Dynavec that takes an argument to pass to the
  // constructor.  If we make the flavor a subclass, then we get into a virtual
  // method call for every extendTo on a dynavec, which is heavier than what I
  // had in mind (but not necessarily all that bad; extendTo is inherently
  // slow, since it often does allocation, so none of the inner loops I care
  // about do many extendTo's).  If we don't make the new Dynavec a subclass,
  // then it's incompatible with all of the extisting utility functions that do
  // useful stuff with Dynavec's.  If we just add another method to the
  // existing Dynavec class, then the existing utility stuff will work unless
  // it calls extendTo without the extra argument for the constructor.
  // 1.5. Hey, if the subclass actually makes a Dynavec of
  // SP<ButtonSlotValue>'s, the problem goes away, since it would be
  // initialized to null pointers without any need to call BSV's constructor.
  // 2. let the person who wants to make a dynavec of ButtonSlotValue's to take
  // a subclass of ButtonSlotValue that constructs and provides the event that
  // ButtonSlotValue requires.  Perhaps this is best.  I can imagine situations
  // where it would leave the class that wants to make a dynavec of
  // ButtonSlotValue's in the same situation as ButtonSlotValue itself, but
  // maybe that's just imagining more than is useful.
  // 3. We could rearrange things so that ButtonSlotValue
  // is a template and the methods that BSV wants to use on Event are all
  // static.  This last option isn't very attractive because it implies that
  // lots of other things that use ButtonSlotValue or the template-requiring
  // Events would have to be templates too, and I think separate compilation is
  // a good thing.
  // 4. Maybe we never will need a dynavec of ButtonSlotValue's.
  ButtonSlotValue (SP<const Event> e);
  ButtonSlotValue (SP<const Event> e, Event::Button b);
  ~ButtonSlotValue ();
  ButtonSlotValue (const ButtonSlotValue &bsv);
  Event::Button getValue () const;
  void setValue (Event::Button button);
  String unParse () const;
  void editSlotValue ();
  SP<SlotValue> copy () const;
  bool hasTheSameTypeAs (const SlotValue *s) const;
  static Event::Button getSlot (const Configuration *a_c,
				const String &slotName);
  static void setSlot (Configuration *a_c,
		       const String &slotName,
		       Event::Button b);
};
#endif
