// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __ButtonSlotValue_h__
#include "ButtonSlotValue.h"
#endif

#ifndef __Event_h__
#include "Event.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __StreamUtil_h__
#include "StreamUtil.h"
#endif

#ifndef __BadSlotValue_h__
#include "BadSlotValue.h"
#endif

#ifndef __Configuration_h__
#include "Configuration.h"
#endif

#ifndef __SlotValue_h__
#include "SlotValue.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __iostream_h__
#include <iostream.h>
#define __iostream_h__
#endif

struct ButtonSlotValue::ButtonSlotValueData {
  Event::Button m_value;
  SP<const Event> m_event;
  ButtonSlotValueData (SP<const Event> e)
    : m_value (e->minButton ()),
      m_event (e)
  {}
  inline void check () {
#ifndef NDEBUG
    assert (m_event->isValidButton (m_value));
#endif
  }
};

ButtonSlotValue::ButtonSlotValue (const ButtonSlotValue &bsv)
  : SlotValue (bsv),
    m_data (NEW (ButtonSlotValueData (bsv.m_data->m_event)))
{
  *m_data = *(bsv.m_data);
  m_data->check ();
}

ButtonSlotValue::ButtonSlotValue (SP<const Event> e)
  : SlotValue ("Button"),
    m_data (NEW (ButtonSlotValueData (e)))
{
  m_data->check ();
}

ButtonSlotValue::ButtonSlotValue (SP<const Event> e, Event::Button b)
  : SlotValue ("Button"),
    m_data (NEW (ButtonSlotValueData (e)))
{
  m_data->m_value = b;
  m_data->check ();
}

ButtonSlotValue::~ButtonSlotValue ()
{
  assert (m_data);
  delete m_data;
  m_data = 0;
}

Event::Button ButtonSlotValue::getValue () const {
  m_data->check ();
  return m_data->m_value;
}

void ButtonSlotValue::setValue (Event::Button button) {
  m_data->m_value = button;
  m_data->check ();
}

String ButtonSlotValue::unParse () const {
  return m_data->m_event->unParseButton (m_data->m_value);
}

void ButtonSlotValue::editSlotValue () {
  cout << "The old value of the button is " << unParse () << "." << endl;
  for (;;) {
    String buf;
    StreamUtil::promptRead ("What do you want the new value to be? ",
			    buf);
    const Event::Button parsed = m_data->m_event->parseButton (buf);
    const Event::Button first = m_data->m_event->minButton ();
    const Event::Button last = m_data->m_event->maxButton ();
    if (last == parsed) {
      cout << "You must specify the name of a button, which is a " << endl
	   << "printable non-upper-case character or one of the " << endl
	   << "following:" << endl;
      int column = 0;
      for (Event::Button i = first; i < last; i++) {
	if (m_data->m_event->isValidButton (i)) {
	  if (i != first) {
	    cout << ", ";
	    column += 2;
	  }
	  const String unParsed = m_data->m_event->unParseButton (i);
	  if (column + unParsed.size() + 2 > 75) {
	    cout << endl;
	    column = 0;
	  }
	  cout << unParsed;
	  column += unParsed.size();
	}
      }
      cout << endl;
    } else {
      m_data->m_value = parsed;
      break;
    }
  }
}

bool ButtonSlotValue::hasTheSameTypeAs (const SlotValue *s) const {
  return 0 != dynamic_cast <const ButtonSlotValue *> (s);
}

SP<SlotValue> ButtonSlotValue::copy () const {
  return NEW (ButtonSlotValue (*this));
}

Event::Button ButtonSlotValue::getSlot (const Configuration *a_c,
				     const String &slotName) {
  SP<const Configuration> c = a_c;
  SP<const SlotValue> slot = c->getSlot (slotName);
  const ButtonSlotValue *islot =
    dynamic_cast <const ButtonSlotValue *> (&*slot);
  assert (islot);
  return islot->getValue();
}

void ButtonSlotValue::setSlot (Configuration *a_c,
			       const String &slotName,
			       Event::Button b) {
  SP<Configuration> c = a_c;
  SP<ButtonSlotValue> bsv =
    dynamic_cast <ButtonSlotValue *> (&*c->getSlot (slotName));
  assert (bsv);
  SP<BadSlotValue> bad =
    c->setSlot (slotName, NEW (ButtonSlotValue (bsv->m_data->m_event, b)));
  assert (!bad);
}
