// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#include "HashTableImpl.h"

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __stdlib_h__
#include <stdlib.h>
#define __stdlib_h__
#endif

#ifndef __HashTableIteratorTypeless_h__
#include "HashTableIteratorTypeless.h"
#endif

#ifndef __HashTableTypeless_h__
#include "HashTableTypeless.h"
#endif

typedef Hashable::HASH HASH;

struct HashTableEnt {
  SP <const Hashable> key;
  HASH hash;
  SP <Refcount> value;
};

typedef HashTableEnt *TABLEENT;

static TABLEENT table_new_vec (int sizeShift) {
  int size = 1 << sizeShift;
  TABLEENT result = NEW (HashTableEnt [size]);
  return result;
}

HashTableImpl::HashTableImpl () {
  sizeShift = INITIAL_SIZE_SHIFT;
  vec = table_new_vec (sizeShift);
  currentUsage = 0;
  figureMaxUsage ();
}

#ifndef NDEBUG
int HashTableImpl::growths = 0;
int HashTableImpl::collisions = 0;
int HashTableImpl::probes = 0;
#endif

HashTableImpl::~HashTableImpl ()
{
  delete [] vec;
}

static size_t trimHash (HASH thinghash, int size)
{
  return thinghash & ((1 << size) - 1);
}

void HashTableImpl::figureMaxUsage ()
{
  /* Maximum usage is 3/4 * the size. */
  maxUsage = (1 << (sizeShift - 2)) * 3;
}

/* Find a slot for key in the table.  There had better be an empty */
/* slot.  Either return a suitable empty slot for adding a */
/* new entry with the given key, or a slot that already has the key, */
/* if there is one. */
int HashTableImpl::table_slot (const Hashable &key, HASH thisHash) const
{
  size_t thisSlot = trimHash (thisHash, sizeShift);
#ifndef NDEBUG
  /* Next one is only used in the assert below. */
  size_t startSlot = thisSlot;
#endif
  size_t mask = (1 << sizeShift) - 1;
  TABLEENT entries = vec;
#ifndef NDEBUG
  probes++;
#endif
  for (;;) {
    TABLEENT thisEntry = &(entries [thisSlot]);
    if (! thisEntry->key) return thisSlot;
    if (thisEntry->hash == thisHash &&
	// Using operator== for Hashable here.
	*(thisEntry->key) == key)
      return thisSlot;
#ifndef NDEBUG
    collisions++;
#endif
    thisSlot++;
    thisSlot &= mask;
    /* If startSlot is equal to thisSlot, then we filled up the table */
    /* without growing it. */
    assert (startSlot != thisSlot);
  }
}

/* Potential optimization - when we search for the table slot while */
/* rehashing, we don't have to do any string comparisons because we */
/* know they'll all be diffent. */
void HashTableImpl::growTable ()
{
  TABLEENT oldVec = vec;
  TABLEENT oldPtr, newPtr;
  size_t oldSize = 1 << sizeShift;
  unsigned int i;
#ifndef NDEBUG
  growths++;
#endif
  sizeShift += 2;
  vec = table_new_vec (sizeShift);
  for (i = 0, oldPtr = oldVec; i < oldSize; i++, oldPtr++) {
    if (oldPtr->key) {
      int slot = table_slot (*(oldPtr->key), oldPtr->hash);
      newPtr = &(vec[slot]);
      assert (! newPtr->key);
      newPtr->key = oldPtr->key;
      newPtr->hash = oldPtr->hash;
      newPtr->value = oldPtr->value;
    }
  }
  delete [] oldVec;
  figureMaxUsage ();
}

void HashTableImpl::store (const Hashable &key, SP<Refcount> value)
{
  HASH thisHash = key.hash();
  int slot = table_slot (key, thisHash);
  TABLEENT ent = &(vec[slot]);
  if (ent->key) {
    /* Smart pointers take care of destructing the old value. */
    ent->value = value;
  } else {
    /* A new value.  Fill in the hash. */
    ent->key = key.copy();
    ent->value = value;
    ent->hash = thisHash;
    currentUsage++;
    if (currentUsage > maxUsage) {
      growTable ();
      /* The entry may have moved if we rehashed. */
      // But we don't care since we don't need to know the slot in
      // this version.
      // slot = table_slot (key, thisHash);
      // ent = &(vec[slot]);
    }
  }
  // return ent->key;
}

Refcount *HashTableImpl::load (const Hashable &key)
{
  HASH thisHash = key.hash();
  int slot = table_slot (key, thisHash);
  if (vec[slot].key) {
    TABLEENT entry = &(vec[slot]);
    return entry->value;
  } else {
    return 0;
  }
}

const Refcount *HashTableImpl::load (const Hashable &key) const
{
  HASH thisHash = key.hash();
  int slot = table_slot (key, thisHash);
  if (vec[slot].key) {
    const TABLEENT entry = &(vec[slot]);
    return entry->value;
  } else {
    return 0;
  }
}

HashTableIteratorTypeless HashTableImpl::iterator () {
  return HashTableIteratorTypeless ((HashTableTypeless *)this);
}

void HashTableImpl::iteratorNormalize (HashTableIteratorTypeless &i) const {
  const int size = 1 << sizeShift;
  int ind = i.m_index;
  for (;;) {
    if (ind >= size) {
      break;
    } else if (vec[ind].key) {
      break;
    }
    ind++;
  }
  i.m_index = ind;
}

void HashTableImpl::iteratorFirst (HashTableIteratorTypeless &i) const
{
  i.m_index = 0;
  iteratorNormalize (i);
}

void HashTableImpl::iteratorNext (HashTableIteratorTypeless &i) const
{
  assert (! iteratorIsDone (i));
  i.m_index++;
  iteratorNormalize (i);
}

bool HashTableImpl::iteratorIsDone (const HashTableIteratorTypeless &i) const
{
  return i.m_index >= (1 << sizeShift);
}

const Hashable *HashTableImpl::iteratorCurrentItem (const HashTableIteratorTypeless &i) const
{
  assert (i.m_index < (1 << sizeShift));
  const Hashable *result = vec[i.m_index].key;
  assert (result);
  return result;
}
  
