// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __Hashtable_h__
#define __Hashtable_h__

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __HashTableTypeless_h__
#include "HashTableTypeless.h"
#endif

#ifndef __HashTableFactory_h__
#include "HashTableFactory.h"
#endif

#ifndef __HashTableIteratorTypeless_h__
#include "HashTableIteratorTypeless.h"
#endif

#ifndef __Refcount_h__
#include "Refcount.h"
#endif

// A template, for type safety, that delegates everything to the
// type-unsafe HashTableTypeless class, which is an ABC that insulates
// us from changes to the HashTableImpl class.
template <class DOMAIN, class RANGE> class HashTable : public Refcount {
  SP<HashTableTypeless> m_table;
 public:
  // This iterates the keys.  You can get the values with "load".
  class HashTableIterator {
    HashTableIteratorTypeless m_iterator;
    friend class HashTable;
    inline HashTableIterator (const HashTableIteratorTypeless &i) :
      m_iterator (i) {}
  public:
    inline void first () {
      m_iterator.first ();
    }
    inline void next () {
      m_iterator.next ();
    }
    inline bool isDone () const {
      return m_iterator.isDone ();
    }
    inline DOMAIN &currentItem () const {
      return *((DOMAIN *) m_iterator.currentItem ());
    }
    // Default assignment, copy, destructor are good.
  };
  inline HashTable () {
    m_table = HashTableFactory::mkHashTableTypeless ();
  }
  inline void store (const DOMAIN &d, SP<RANGE> r) {
    m_table->store (static_cast <const Hashable &> (d),
		    static_cast <Refcount *> (r));
  }
  // Don't need load to return a smart pointer, since the hash table
  // retains a pointer to the thing that was loaded.  Null is returned
  // if nothing was found.
  inline RANGE *load (const DOMAIN &d) {
    Refcount *result =
      m_table->load (static_cast <const Hashable &> (d));
    assert (! result || (dynamic_cast <RANGE *> (result)));
    return dynamic_cast <RANGE *> (result);
  }
  inline const RANGE *load (const DOMAIN &d) const {
    const Refcount *result =
      m_table->load (static_cast <const Hashable &> (d));
    assert (! result || (dynamic_cast <const RANGE *> (result)));
    return static_cast <const RANGE *> (result);
  }
  inline HashTableIterator iterator () {
    return HashTableIterator (m_table->iterator());
  }
};

#endif
