// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#include "HashTable.h"

// Nothing here but a test case.  It uses String, which is in another
// package.
#ifdef UNIT_TEST
// Compilation command:
// g++ -Wall -Wwrite-strings -Werror -pipe -felide-constructors -g  -I/home/tim/fungimol/Util -I/home/tim/fungimol/String -I. -DUNIT_TEST /home/tim/fungimol/Hashtable/HashTable.cpp /home/tim/fungimol/String/*.o /home/tim/fungimol/Hashtable/*.o /home/tim/fungimol/Util/*.o -o test

#ifdef NDEBUG
#warn This test case makes no sense if compiled with -DNDEBUG.
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __stdio_h__
#include <stdio.h>
#define __stdio_h__
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#define MAXLEN 200

#ifndef __Refcount_h__
#include "Refcount.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

// Need next one for growths, probes, and collisions.
#ifndef __HashTableImpl_h__
#include "HashTableImpl.h"
#endif

class RefcountedInteger : public Refcount {
  int m_int;
public:
  RefcountedInteger (int i) : m_int (i) {}
  int getInt () const { return m_int; }
  void setInt (int i) { m_int = i; }
  operator int () const { return m_int; }
};

int main (int argc, char **argv)
{
  char *filename;
  FILE *fp;
  if (2 != argc) {
    printf ("Should pass exactly one file argument, for now.\n");
    exit (1);
  }
  filename = argv [1];
  fp = fopen (filename, "r");
  if (0 == fp) {
    printf ("Failed to open %s.\n", filename);
    exit (2);
  }
  {
    HashTable <String, RefcountedInteger> tab;
    int lineTotal = 0;
    int repeatedLineTotal = 0;
    
    /* First time, make a table mapping the text of each line in */
    /* the file to the line number.  If the line is repeated, the */
    /* first line number goes in the table.  lineTotal is the sum */
    /* of the line numbers in the table and repeatedLineTotal is */
    /* the sum of the line numbers of the repetitions of the */
    /* lines. */
    char buf [MAXLEN];
    char *cp;
    {
      int lineno = 0;
      for (;;) {
	cp = fgets (buf, MAXLEN, fp);
	if (0 == cp) break;
	lineno++;
	if (tab.load (String(buf))) {
	  repeatedLineTotal += lineno;
	} else {
	  lineTotal += lineno;
	  tab.store (buf, NEW (RefcountedInteger (lineno)));
	}
      }
      printf ("Finished storing %d lines.\n", lineno);
    }
    rewind (fp);
    {
      int lineno = 0;
      int lineTotal2 = 0;
      for (;;) {
	SP<RefcountedInteger> oldLine;
	cp = fgets (buf, MAXLEN, fp);
	if (0 == cp) break;
	lineno++;
	oldLine = tab.load (buf);
	assert (oldLine);
	assert ((*oldLine) <= lineno);
	if ((*oldLine) == lineno) {
	  lineTotal2 += lineno;
	} else {
	  repeatedLineTotal -= lineno;
	}
      }
      assert (lineTotal2 == lineTotal);
      assert (0 == repeatedLineTotal);
    }
    {
      int stat = fclose (fp);
      if (0 != stat) {
	printf ("Failed to close %s.\n", filename);
	exit (3);
      }
    }
    HashTable <String, RefcountedInteger>::HashTableIterator i =
      tab.iterator ();
    for (;;) {
      if (i.isDone()) break;
      lineTotal -= *(tab.load (i.currentItem()));
      i.next ();
    }
    assert (0 == lineTotal);
  }
  printf ("%d growths; %d probes; %d collisions\n",
	  HashTableImpl::growths,
	  HashTableImpl::probes,
	  HashTableImpl::collisions);
  listLeaks ();
  return 0;
}
#endif
