// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#include "Geometry.h"

#ifndef __math_h__
#include <math.h>
#define __math_h__
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

inline bool aboutEqual (Float f1, Float f2) {
  Float diff = f1 - f2;
  if (diff < 0) diff = -diff;
  return diff < 0.001;
}

// We have a triangle in two dimensions.  The points are the origin O
// at (0,0), point A at (sOA, 0), and point B.  The sides are sOA,
// sOB, sAB.  Return the coordinates of B.
// sOAsOA is the square of sOA, etc.  We take squares as arguments
// because that way we don't force the caller to do square roots that
// aren't needed.
void Geometry::findTrianglePoint (Float sOAsOA, Float sOBsOB, Float sABsAB,
				  Float sOA,
				  Float & Bx, Float & By) {
  // Let point C = (Bx, 0).
  // Pythagoras on sOB gives:
  // sqr(sOB) = sqr(Bx) + sqr(By), thus
  // sqr(By) = sqr(sOB) - sqr(Bx).   (Equation 1)
  // Pythagoras on sAB gives:
  // sqr(sAB) = sqr(sOA - Bx) + sqr(By), which multiplies out to
  // sqr(sAB) = sqr(sOA) - 2 sOA Bx + sqr(Bx) + sqr(By)
  // and putting sqr(By) by itself gives
  // sqr(By) = sqr(sAB) - sqr(sOA) + 2 sOA Bx - sqr(Bx).  (Equation 2)
  // Setting the rhs's of Equations 1 and 2 equal gives
  // sqr(sOB) - sqr(Bx) = sqr(sAB) - sqr(sOA) + 2 sOA Bx - sqr(Bx)
  // sqr(sOB) = sqr(sAB) - sqr(sOA) + 2 sOA Bx
  // 2 sOA Bx = sqr(sOB) - sqr(sAB) + sqr(sOA)
  // Bx = (sqr(sOB) - sqr(sAB) + sqr(sOA)) / (2 * sOA)
  // Example: sOA = 3, sOB = 5, sAB = 4, should give Bx = 3.
  // Example: sOA = 3, sOB = 4, sAB = 5, should give Bx = 0.

  assert (sOA > 0);
  Bx = (sOBsOB - sABsAB + sOAsOA) / (2 * sOA);

  // Once we have Bx, Pythagoras tells us By:
  // sqr(Bx)  + sqr(By) = sqr(sOB), so By = sqrt (sqr(sOB) - sqr(Bx)).
  // Also,
  // sqr(sAC) + sqr(By) = sqr(sAB), so By = sqrt (sqr(sAB) - sqr(sAC)).
  // Subtracting two nearly-equal floats is less accurate than
  // subtracting two not-very-equal floats.  Since the difference is
  // constant, we prefer the difference of the smaller numbers, so we
  // should take the first formula if sOB is smaller than 
  // sOA, otherwise take the second.
  if (sOBsOB < sABsAB) {
    By = sqrt (sOBsOB - Bx * Bx);
  } else {
    const Float sAC = sOA - Bx;
    By = sqrt (sABsAB - sAC * sAC);
  }
  assert (aboutEqual (Bx * Bx + By * By, sOBsOB));
  assert (aboutEqual ((sOA - Bx) * (sOA - Bx) + By * By, sABsAB));
}

void Geometry::findTrianglePoint (Float sOA, Float sOB, Float sAB,
				  Float & Bx, Float & By) {
  findTrianglePoint (sOA * sOA, sOB * sOB, sAB * sAB, sOA, Bx, By);
}

#ifdef UNIT_TEST
// Compilation command:
// g++ -Wall -Winline -W -Wwrite-strings -Werror -pipe -felide-constructors -g -fno-exceptions  -I/home/tim/fungimol/LinearAlgebra -I/home/tim/fungimol/Util /home/tim/fungimol/Graphics/Geometry.cpp /home/tim/fungimol/Util/debug/myassert.o -o /tmp/a.out -DUNIT_TEST

#ifndef __iostream_h__
#include <iostream.h>
#define __iostream_h__
#endif

int main () {
  Float f1;
  Float f2;
  Geometry::findTrianglePoint (3, 4, 5, f1, f2);
  Geometry::findTrianglePoint (5, 10, 7, f1, f2);
  Geometry::findTrianglePoint (5, 10, 14, f1, f2);
  Geometry::findTrianglePoint (14, 10, 5, f1, f2);
  Geometry::findTrianglePoint (5, 5, 0, f1, f2);
  Geometry::findTrianglePoint (5, 0, 5, f1, f2);
  Geometry::findTrianglePoint (5, 5, 10, f1, f2);
  Geometry::findTrianglePoint (5, 10, 5, f1, f2);
  Geometry::findTrianglePoint (10, 5, 5, f1, f2);
  return 0;
}

#endif
