// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#include "BoundingSphere.h"

#ifndef __assert_h__
#include "assert.h"
#define __assert_h__
#endif

#ifndef __FloatUtil_h__
#include "FloatUtil.h"
#endif

#ifndef __BoundingBox_h__
#include "BoundingBox.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

BoundingSphere::BoundingSphere (const BoundingBox &box) {
  Vec3 least = box.least();
  Vec3 most = box.most();
  Float size = FloatUtil::max (most[0]-least[0],
			       FloatUtil::max (most [1] - least [1],
					       most [2] - least [2]));
  m_center = (least + most) / 2;
  static const Float root2per2 = FloatUtil::sqrt (2.0) / 2.0;
  m_radius = size * root2per2;
}

bool BoundingSphere::overlaps (const BoundingSphere &s) const {
  if (!s.isBounded() || !isBounded()) {
    return 1;
  } else {
    Float radiusSum = radius () + s.radius ();
    Float rSquared = radiusSum * radiusSum;
    Float distSquared = (center() - s.center ()).lengthSquared();
    return distSquared <= rSquared;
  }
}

BoundingSphere::BoundingSphere (Float a_radius, const Vec3 &a_center)
  : m_radius (a_radius), m_center (a_center)
{}

BoundingSphere::BoundingSphere () : m_radius (-1.0)
#ifndef NDEBUG
     // Initialize to silly values to get deterministic bugs.
     , m_center (-1000.0, -1000.0, -1000.0)
#endif     
{}

void BoundingSphere::merge (const BoundingSphere &toAdd) {
  if (!toAdd.isBounded ()) {
    // Done.
  } else if (!isBounded ()) {
    m_radius = toAdd.radius ();
    m_center = toAdd.center ();
  } else {
    Vec3 toNew = toAdd.center () - center ();
    // FIXME Bounding Spheres are silly because they need square roots
    // We can do Bounding rectangles or cubes without needing square
    // roots.  We can do even better with octtrees; just take the
    // bounds of the toplevel node.
    // Deferring the fix because this code works fine; it's just slower than it
    // could be if we used the alternatives.
    Float distance = toNew.length ();
    if (distance + toAdd.radius() <= radius ()) {
      // toAdd is enclosed within this, so we're done.
    } else if (0.0001 >= distance) {
      // They have the same center, and the old one doesn't enclose
      // the new one, so the new one encloses the old one.
      m_radius = toAdd.radius ();
    } else {
      Vec3 toNewUnit = toNew / distance;
      Float newRadius = (radius () + distance + toAdd.radius ()) / 2;
      m_center += toNewUnit * (newRadius - radius());
      m_radius = newRadius;
    }
  }
  assert (! FloatUtil::isNAN (m_radius));
}

bool BoundingSphere::contains (const Vec3 &v) const {
  if (isBounded ()) {
    return (v - m_center).lengthSquared () < FloatUtil::sqr (m_radius);
  } else {
    return true;
  }
}


