// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __BoundingBoxInt_h__
#define __BoundingBoxInt_h__

#ifndef __Float_H__
#include "Float.h"
#endif

#ifndef __Vec3i_h__
#include "Vec3i.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

class BoundingBox;

// Each coordinate of m_min should be less than or equal to the
// corresponding coordinate in m_max.  The box includes m_min but it
// excludes m_max.
class BoundingBoxInt {
  // Next two are really const, but declaring them const would require
  // redundant code in the constructor.
  Vec3i m_min;
  Vec3i m_max;
 public:
  inline BoundingBoxInt () {};
  // Return a BoundingBoxInt that includes the given b, where one unit
  // in the result is unitSize in b.
  BoundingBoxInt (const BoundingBox &b, Float unitSize);
  // Return a BoundingBoxInt that has v as the minimum and is a size x
  // size x size cube.
  BoundingBoxInt (const Vec3i &v, int size);
  inline BoundingBoxInt (const Vec3i &min, const Vec3i &max) : m_min (min), m_max (max)
    {};
 private:
  static inline bool intervalOverlap (int i1min, int i1max,
				      int i2min, int i2max) {
    assert (i1min <= i1max);
    assert (i2min <= i2max);
    return (i2min < i1max) && (i1min < i2max);
  }
 public:
  inline bool overlaps (const BoundingBoxInt &b) const {
    return (intervalOverlap (m_min [0], m_max [0], b.m_min [0], b.m_max [0]) &&
	    intervalOverlap (m_min [1], m_max [1], b.m_min [1], b.m_max [1]) &&
	    intervalOverlap (m_min [2], m_max [2], b.m_min [2], b.m_max [2]));
  }
 private:
  static inline bool intervalContains (int i1min, int i1max,
				       int i2min, int i2max) {
    assert (i1min <= i1max);
    assert (i2min <= i2max);
    return ((i1min <= i2min) && (i2max <= i1max));
  }
 public:
  inline bool contains (const BoundingBoxInt &b) const {
    return (intervalContains (m_min [0], m_max [0], b.m_min [0], b.m_max [0]) &&
	    intervalContains (m_min [1], m_max [1], b.m_min [1], b.m_max [1]) &&
	    intervalContains (m_min [2], m_max [2], b.m_min [2], b.m_max [2]));
  }
  inline bool operator== (const BoundingBoxInt &b) const {
    return (m_min == b.m_min) && (m_max == b.m_max);
  }
  inline Vec3i min () const { return m_min; }
  inline Vec3i max () const { return m_max; }
};

#endif
