// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __BoundingBox_h__
#define __BoundingBox_h__

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

class BoundingSphere;
class BoundingBoxInt;

// Unlike BoundingSphere's, BoundingBox's can't represent the
// boundaries of an unbounded object.
class BoundingBox {
  // We should have m_less <= m_more, comparing coordinate-by-coordinate.
  // The box is inclusive of m_less and exclusive of m_more, although
  // that doesn't matter much since Vec3's are hardly ever exactly
  // equal to each other.  This convention makes it easy to convert to
  // BoundingBoxInt's.  See also the definition of BoundingSphere.h.
  Vec3 m_less;
  Vec3 m_more;
 public:
  // Next one should only be called when the sphere is bounded.
  BoundingBox (const BoundingSphere &bs);
  BoundingBox (const BoundingBoxInt &bbi, const Float unitSize);
  BoundingBox (const Vec3 &less, const Vec3 &more)
    : m_less (less), m_more (more)
  {}
  inline BoundingBox () {};
  inline Vec3 least () const {return m_less;};
  inline Vec3 most () const {return m_more;};
 private:
  static inline bool intervalOverlap (Float i1min, Float i1max,
				      Float i2min, Float i2max) {
    assert (i1min <= i1max);
    assert (i2min <= i2max);
    return (i2min < i1max) && (i1min < i2max);
  }
 public:
  inline bool overlaps (const BoundingBox &b) const {
    return (intervalOverlap (m_less [0], m_more [0], b.m_less [0], b.m_more [0]) &&
	    intervalOverlap (m_less [1], m_more [1], b.m_less [1], b.m_more [1]) &&
	    intervalOverlap (m_less [2], m_more [2], b.m_less [2], b.m_more [2]));
  }
 private:
  static inline bool intervalContains (Float i1min, Float i1max,
				       Float i2min, Float i2max) {
    assert (i1min <= i1max);
    assert (i2min <= i2max);
    return ((i1min <= i2min) && (i2max <= i1max));
  }
 public:
  inline bool contains (const BoundingBox &b) const {
    return (intervalContains (m_less [0], m_more [0], b.m_less [0], b.m_more [0]) &&
	    intervalContains (m_less [1], m_more [1], b.m_less [1], b.m_more [1]) &&
	    intervalContains (m_less [2], m_more [2], b.m_less [2], b.m_more [2]));
  }
  inline bool contains (const Vec3 &b) const {
    return (m_less [0] <= b[0] && b[0] < m_more [0] &&
	    m_less [1] <= b[1] && b[1] < m_more [1] &&
	    m_less [2] <= b[2] && b[2] < m_more [2]);
  }
  // Make the box smaller in each directions by radius.  That is, the
  // center of the new box equals the center of the old box, and the
  // length of the new box is 2 * radius less than the length of the
  // old box, in each direction.
  inline BoundingBox shrink (Float radius) {
    Vec3 rVec (radius, radius, radius);
    return BoundingBox (m_less + rVec, m_more - rVec);
  }
  // Given a corner number 0 to 7, return the coordinates of that
  // corner of the cube.  The numbering scheme is arbitrary.
  Vec3 corner (int which) const;
  // Nearest point to v that's in the box, or at any rate on the closure of the
  // box.
  Vec3 nearest (const Vec3 &v);
  bool intersects (const BoundingSphere &bs);
};

inline bool operator== (const BoundingBox &b1, const BoundingBox &b2) {
  return b1.least() == b2.least() && b1.most() == b2.most();
}
#endif
