// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#include "BoundingBox.h"

#ifndef __BoudingSphere_h__
#include "BoundingSphere.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif 

#ifndef __FloatUtil_h__
#include "FloatUtil.h"
#endif

BoundingBox::BoundingBox (const BoundingSphere &bs) {
  assert (bs.isBounded ());
  Vec3 center = bs.center ();
  Float radius = bs.radius ();
  assert (radius >= 0);
  m_less = Vec3 (center [0] - radius, center [1] - radius, center [2] - radius);
  m_more = Vec3 (center [0] + radius, center [1] + radius, center [2] + radius);
}

#ifndef __BoundingBoxInt_h__
#include "BoundingBoxInt.h"
#endif

#ifndef __Vec3i_h__
#include "Vec3i.h"
#endif

BoundingBox::BoundingBox (const BoundingBoxInt &bbi, const Float unitSize) {
  m_less = bbi.min() * unitSize;
  m_more = bbi.max() * unitSize;
}

Vec3 BoundingBox::corner (int which) const
{
  assert (0 <= which);
  assert (which < 8);
  return Vec3 ((which & 1) ? m_less[0]:m_more[0],
	       (which & 2) ? m_less[1]:m_more[1],
	       (which & 4) ? m_less[2]:m_more[2]);
}

Vec3 BoundingBox::nearest (const Vec3 &v) {
  Vec3 nearV;
  for (int i = 0; i < 3; i++) {
    nearV [i] = FloatUtil::max (m_less [i],
				FloatUtil::min (m_more[i], v[i]));
  }
  return nearV;
}

bool BoundingBox::intersects (const BoundingSphere &bs) {
  return bs.contains (nearest (bs.center()));
}  

