// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __CursorBall_h__
#include "CursorBall.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __Quaternion_h__
#include "Quaternion.h"
#endif

#ifndef __SceneGraph_h__
#include "SceneGraph.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __Configuration_h__
#include "Configuration.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __XYZCursorBallConfiguration_h__
#include "XYZCursorBallConfiguration.h"
#endif

#ifndef __Factory_h__
#include "Factory.h"
#endif

#ifndef __Configurable_h__
#include "Configurable.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __ConstantRecursiveSlotValue_h__
#include "ConstantRecursiveSlotValue.h"
#endif

namespace {
  class XYZCursorBall
    : public CursorBall
  {
    mutable OL<CursorBall> m_axisLinks [3];
    // We save the markers themselves, instead of only saving their positions,
    // so that if they get deleted we can put them back.
    mutable SP<CursorBall> m_axisMarkers [3];
    mutable SP<XYZCursorBallConfiguration> m_conf;
  public:
    XYZCursorBall (XYZCursorBallConfiguration *conf)
      : CursorBall (conf),
	m_conf (conf)
    {}
    XYZCursorBall (const XYZCursorBall &cb)
      : CursorBall (cb),
	m_conf (cb.m_conf)
    {
      for (int i = 0; i < 3; i++) {
	m_axisMarkers [i] = cb.m_axisMarkers [i]->copy ();
	// Let m_axisLinks [i] stay invalid.
      }
    }
    SP<CursorBall> copy () const {
      return NEW (XYZCursorBall (*this));
    }
    OL<CursorBall> insert (SceneGraph *sg) const {
      for (int i = 0; i < 3; i++) {
	SP<ConstantRecursiveSlotValue> crsv = m_conf->getAxisMarker ('X'+i);
	SP<Configurable> c = crsv->makeIt ();
	assert (c);
	m_axisMarkers [i] = dynamic_cast <CursorBall *> (&*c);
	assert (m_axisMarkers [i]);
	m_axisLinks [i] = m_axisMarkers [i]->insert (sg);
      }
      OL<CursorBall> result = CursorBall::insert (sg);
      // Need to do a moveTo so the initial cursorball gets its axis markers.
      moveTo (result, result->getPosition(), result->getRotation());
      return result;
    }
    void moveTo (OL<CursorBall> &cbl, const Vec3 &where, const Quaternion &rot)
      const
    {
      for (int i = 0; i < 3; i++) {
	Vec3 v (0, 0, 0);
	v[i] = 1;
	// If they deleted any of the link objects, this should do the right
	// thing anyway.
	m_axisMarkers [i]->moveTo (m_axisLinks [i], where + rot.rotate (v),
				   rot);
      }
      CursorBall::moveTo (cbl, where, rot);
    }
    ~XYZCursorBall () {
      for (int i = 0; i < 3; i++) {
	if (m_axisLinks[i].isValid()) {
	  m_axisLinks[i].deleteObject ();
	}
      }	
    }
  };
  class XYZCursorBallFactory
    : public Factory
  {
  public:
    XYZCursorBallFactory ()
      : Factory ("XYZCursorBall")
    {}
    SP<Configuration> defaultConfiguration () const {
      return NEW (XYZCursorBallConfiguration ());
    }
    SP<Configurable> makeIt (SP<Configuration> c) const {
      SP<XYZCursorBallConfiguration> conf =
	dynamic_cast <XYZCursorBallConfiguration *> (&*c);
      assert (conf);
      return NEW (XYZCursorBall (conf));
    }
  };
  const bool useless =
  (FactoryTable::store ("CursorBall", NEW (XYZCursorBallFactory ())),
   true);
};
