// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __CursorBall_h__
#define __CursorBall_h__

#ifndef __BoringBall_h__
#include "BoringBall.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __OL_h__
#include "OL.h"
#endif

class Vec3;
class Quaternion;
class Refcount;
class SceneGraph;
class CursorBallConfiguration;

class CursorBall
  : public BoringBall
{
  struct CursorBallData;
  CursorBallData *m_data;
public:
  CursorBall (SP<CursorBallConfiguration> conf);
  ~CursorBall ();
  // Typical use of a cursor ball:
  // A command is bound to the motion of some pointer device.  It creates one
  // or more CursorBalls, and distinct Refcount'ed pieces of information to
  // uniquely identify them.  When things happen, the command can change the
  // positions of the cursorballs or change the info stored in them for future
  // reference.  The cursorballs themselves are stored in the configuration of
  // the command in slots that are not visible to the configuration editor.
  //
  // If multiple commands have to be able to share use of some CursorBalls,
  // then you store those CursorBalls in static storage and use
  // MemoryUtil::registerDeallocator to be sure that they don't show up as
  // memory leaks, and one of the factories exports static commands to get the
  // CursorBalls from static storage, or to create them if they aren't already
  // there. 
  //
  // Go find the CursorBall with the given info in the given scene graph.  This
  // is fairly efficient; we only have to do work when the objects in the scene
  // graph get renumbered, or the cursor ball is deleted.  If we can't find the
  // cursor ball, then we enter it into the scene graph and return the new
  // index.  If we do find the cursor ball, we return its existing index in the
  // scene graph.  If we can't find the cursor ball and we can't enter it into
  // the scene graph, then we return null and the index is set to -1.
  //
  // While we're at it, we support the user reconfiguring the size and color of
  // the cursor ball while it's being used.  We compare our size and color to
  // the values specified in the configuration that we were created from, and
  // if those have changed the cursorball that is returned is different from
  // this and has the new color and size, and the old one will be deleted from
  // the scene graph, and the new one will be inserted with the same position
  // as the old one.
  //
  // (The cursor ball points to its configuration and the info.  The scene
  // graph points to the cursor ball.  The configuration points to the info.
  // No reference cycles yet, but if the info points at something interesting
  // that eventually leads off to the configuration of the cursorball, then
  // we'll have a cycle.)
  //
  // If you insert a CursorBall into the scene twice, then moveTo will
  // find one or other other of them, so one of them probably will be ignored.
  // This is not useful.
  //
  // Move the cursorball to the given location.  Updates the configuration.
  // Moving it can cause it to change its identity, if you did a
  // findCursorBall, then changed the radius, then did a move.  So use the
  // result from moveTo.
  //
  // The position of the cursorball is reflected in the configuration.  If you
  // edit the configuration, then the cursorball detects that and will update
  // its position in the scene graph next time findCursorBall is called.
  //
  // If the CursorBall can't be inserted into the scene, we try moving it to
  // location 0,0,0.  If it still can't be inserted into the scene, the link
  // becomes invalid.
  static void updateFromConf (OL<CursorBall> &cbl);
  // Insert the cursor ball into the scene graph.  Has to be virtual because
  // XYZCursorBall wants to do more.
  virtual OL<CursorBall> insert (SceneGraph *sg) const;
  // This just changes the configuration and the scene graph.  It doesn't check
  // that the new location is acceptable to the scene graph.  It doesn't update
  // the size and color from the configuraiton.  We need the cbl so we can
  // update the position in the scene graph.  This has to be a virtual method
  // because, for instance, XYZCursorBall wants to make it do more.
  // This is const because if it has to change the cursorball, it will do it by
  // replacing it with a new one.
  virtual void moveTo (OL<CursorBall> &cbl,
		       const Vec3 &where, const Quaternion &rot) const;
  SP<CursorBallConfiguration> getConf () const;
  // When copying, initialize the non-changeable state from the configuration
  // of the original, not the state of the original.  
  CursorBall (const CursorBall &cb);
  // Copy needs to be virtual, since we'll have subclasses with information
  // that needs to be copied too.
  virtual SP<CursorBall> copy () const;
  Vec3 getPosition () const;
  Quaternion getRotation () const;
  bool isSelectable () const;
};
#endif
