// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __CursorBall_h__
#include "CursorBall.h"
#endif

#ifndef __Refcount_h__
#include "Refcount.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __CursorBallConfiguration_h__
#include "CursorBallConfiguration.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __SceneGraph_h__
#include "SceneGraph.h"
#endif

#ifndef __Factory_h__
#include "Factory.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __Quaternion_h__
#include "Quaternion.h"
#endif

// PhysicsObjects are considered to change when the observable physics or
// graphics change.  None of the data in a CursorBall affects these things, so
// all are mutable.
struct CursorBall::CursorBallData {
  mutable SP<CursorBallConfiguration> m_conf;
  // The last position that we observed in the configuration for this.  If the
  // position in the configuration changes, then the user must have edited the
  // configuration, so move the cursorball.
  mutable Vec3 m_position;
  mutable Quaternion m_rotation;
};

CursorBall::CursorBall (SP<CursorBallConfiguration> conf)
  : BoringBall (&*conf),
    m_data (NEW (CursorBallData ()))
{
  m_data->m_conf = conf;
  m_data->m_position = conf->getPosition ();
}

CursorBall::~CursorBall () {
  assert (m_data);
  delete m_data;
  m_data = 0;
}
  
// Reiterating the header file: copying initializes the unchangeable state from
// the configuration of the original.
CursorBall::CursorBall (const CursorBall &cb)
  : BoringBall (&*cb.m_data->m_conf),
    m_data (NEW (CursorBallData (*cb.m_data)))
{}

SP<CursorBall> CursorBall::copy () const {
  return NEW (CursorBall (*this));
}

Vec3 CursorBall::getPosition () const {
  return m_data->m_position;
}

Quaternion CursorBall::getRotation () const {
  return m_data->m_rotation;
}

SP<CursorBallConfiguration> CursorBall::getConf () const {
  return m_data->m_conf;
}

void CursorBall::updateFromConf (OL<CursorBall> &cbl) {
  SP<CursorBallConfiguration> conf = cbl->getConf ();
  SP<const CursorBall> oldMe = &*cbl;
  if (conf->getMaterial () != oldMe->getMaterial () ||
      conf->getVisibleRadius () != oldMe->visibleRadius ()) {
    // The user edited the configuration.  Replace ourselves.  Next call will
    // create a new object with the appropriate radius and material.
    SP<CursorBall> newMe = oldMe->copy ();
    cbl.deleteObject ();
    cbl = newMe->insert (cbl.getSceneGraph ());
  } else {
    conf->getPosition().copyToPointer (cbl.objectState ());
  }
  cbl->m_data->m_position = conf->getPosition ();
  cbl->m_data->m_rotation = conf->getRotation ();
}

OL<CursorBall> CursorBall::insert (SceneGraph *sg) const {
  OL<CursorBall> result (sg, this, getConf ()->getPosition().pointer());
  if (!result.isValid ()) {
    // The new position is out of bounds, try again at 0,0,0.
    getConf()->setPosition (Vec3 (0, 0, 0));
    m_data->m_position = Vec3 (0, 0, 0);
    result = OL<CursorBall> (sg, this, getConf()->getPosition().pointer());
    // If we can't put a cursorball at 0,0,0, something is broken.
    assert (result.isValid ());
  }
  return result;
}

void CursorBall::moveTo (OL<CursorBall> &cbl,
			 const Vec3 &where,
			 const Quaternion &rotation)
  const
{
  getConf()->setPosition (where);
  getConf()->setRotation (rotation);
  m_data->m_position = where;
  m_data->m_rotation = rotation;
  if (!cbl.isValid()) {
    // If they deleted it and then want to move it, put it back.
    cbl = insert (cbl.getSceneGraph ());
  }
  where.copyToPointer (cbl.objectState ());
}

bool CursorBall::isSelectable () const {
  return false;
}

namespace {
  class CursorBallFactory
    : public Factory
  {
  public:
    CursorBallFactory ()
      : Factory ("CursorBall")
    {}
    SP<Configuration> defaultConfiguration () const {
      return NEW (CursorBallConfiguration ());
    }
    SP<Configurable> makeIt (SP<Configuration> c) const {
      SP<CursorBallConfiguration> cbc =
	dynamic_cast <CursorBallConfiguration *> (&*c);
      assert (cbc);
      return NEW (CursorBall (cbc));
    }
  };
  const bool useless =
  (FactoryTable::store ("CursorBall", NEW (CursorBallFactory ())),
   true);
};
