// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#include "VectorSlotValue.h"

// For tolower.
#ifndef __ctype_h__
#include <ctype.h>
#define __ctype_h__
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __StreamUtil_h__
#include "StreamUtil.h"
#endif

#ifndef __iostream_h__
#include <iostream.h>
#define __iostream_h__
#endif

#ifndef __Configuration_h__
#include "Configuration.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __SlotValue_h__
#include "SlotValue.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __BadSlotValue_h__
#include "BadSlotValue.h"
#endif

namespace {
  Dynavec <SP<SlotValue> >
  copySlotDynavec (const Dynavec <SP<SlotValue> > &v)
  {
    Dynavec <SP<SlotValue> > result;
    result.extendTo (v.size());
    for (int i = 0; i < v.size(); i++) {
      result [i] = v [i]->copy();
    }
    return result;
  }
}

struct VectorSlotValue::VectorSlotValueData {
  Dynavec <SP<SlotValue> > m_vec;
  SP<SlotValue> m_default;
  bool m_constant;
  void printState (ostream &o);
  VectorSlotValueData ()
    : m_constant (false)
  {}
  VectorSlotValueData (const VectorSlotValueData &vsvd) {
    m_vec = copySlotDynavec (vsvd.m_vec);
    m_default = vsvd.m_default->copy();
    m_constant = vsvd.m_constant;
  }
};

VectorSlotValue::VectorSlotValue (SP<SlotValue> def)
  : SlotValue ("Vector of "+def->typeName()),
    m_data (NEW (VectorSlotValueData ()))
{
  m_data->m_default = def;
}

VectorSlotValue::VectorSlotValue (SP<SlotValue> def,
				  const Dynavec <SP<SlotValue> > &values)
  : SlotValue ("Vector of "+def->typeName()),
    m_data (NEW (VectorSlotValueData ()))
{
  m_data->m_vec = values;
  m_data->m_default = def;
}

// Let's try to avoid requiring a default value for ConstantVectorSlotValue's,
// since it will never be used.
String figureName (SP<SlotValue> def,
		   const Dynavec <SP<SlotValue> > &values)
{
  if (values.size () > 0) {
    return "Vector of " + values [0]->typeName();
  } else if (def) {
    return "Vector of " + def->typeName();
  } else {
    return "Vector";
  }
}

VectorSlotValue::VectorSlotValue (SP<SlotValue> def,
				  const Dynavec <SP<SlotValue> > &values,
				  bool constant)
  : SlotValue (figureName (def, values)),
    m_data (NEW (VectorSlotValueData ()))
{
  m_data->m_vec = values;
  m_data->m_default = def;
  m_data->m_constant = constant;
}

VectorSlotValue::~VectorSlotValue () {
  assert (m_data);
  delete m_data;
  m_data = 0;
}

VectorSlotValue::VectorSlotValue (const VectorSlotValue &v)
  : SlotValue (v),
    m_data (NEW (VectorSlotValueData (*(v.m_data))))
{}

bool VectorSlotValue::hasTheSameTypeAs (const SlotValue *s) const {
  SP<const VectorSlotValue> it = dynamic_cast <const VectorSlotValue *> (s);
  return 0 != it &&
    m_data->m_default->hasTheSameTypeAs (it->m_data->m_default);
}

String VectorSlotValue::unParse () const {
  return String("Vector of length ")+m_data->m_vec.size();
}

void VectorSlotValue::VectorSlotValueData::printState (ostream &o) {
  if (!m_constant) {
    cout << "The default value for new elements has the type" << endl
	 << m_default->typeName () << " and the value "
	 << m_default->unParse () << "." << endl;
  }
  cout << "The vector currently has these contents: " << endl;
  Dynavec <Dynavec <String> > tableau;
  tableau.push (Dynavec <String> ());
  tableau[0].push ("Position");
  tableau[0].push ("Value");
  for (int i = 0; i < m_vec.size(); i++) {
    tableau.push (Dynavec <String> ());
    Dynavec <String> &last = tableau[tableau.size()-1];
    last.push (i+1);
    last.push (m_vec[i]->unParse());
  }
  StreamUtil::printTableau (o, tableau);
}

void VectorSlotValue::editSlotValue() {
  for (;;) {
    m_data->printState(cout);
    cout << "These options are available:" << endl;
    {
      Dynavec <Dynavec <String> > tableau;
      tableau.push (Dynavec <String> ("f,0", "Finish editing this vector"));
      if (!m_data->m_constant) {
	tableau.push (Dynavec<String> ("d", "edit the Default value"));
	tableau.push (Dynavec<String>
		      ("i",
		       "Insert a copy of the default value into the vector"));
	tableau.push (Dynavec <String>
		      ("l", "deLete an element from the vector"));
      }
      tableau.push (Dynavec <String> ("e","Edit an element of the vector"));
      tableau.push (Dynavec <String> ("<slot number>",
				      "Edit the given slot of the vector"));
      tableau.push (Dynavec <String> ("p", "Print out the vector again"));
      StreamUtil::printTableau (cout, tableau);
    }
    String line;
    StreamUtil::promptRead ("Which of these do you want to do? ", line);
    char ch = 0 == line.size()?'p':tolower (line[0]);
    if ('f' == ch || '0' == ch) {
      break;
    } else if (!m_data->m_constant && 'd' == ch) {
      SP<SlotValue> oldDefault = m_data->m_default->copy ();
      m_data->m_default->editSlotValue();
      assert (oldDefault->hasTheSameTypeAs (m_data->m_default));
    } else if (!m_data->m_constant && 'i' == ch) {
      cout << "Note that the vector starts with element 1.  Enter 0 if "<<endl
	   << "you have changed your mind and do not want to insert "<<endl
	   << "anything."<<endl;
      int pos;
      StreamUtil::promptRead
	("Where you want the new default element to appear? ", pos,
	 0, m_data->m_vec.size()+2);
      if (0 < pos) {
	pos--;
	m_data->m_vec.extendTo (m_data->m_vec.size()+1);
	for (int i = m_data->m_vec.size()-1; i > pos; i--) {
	  m_data->m_vec[i] = m_data->m_vec[i-1];
	}
	m_data->m_vec[pos] = m_data->m_default->copy();
      }
    } else if ('e' == ch || ('0' <= ch && ch <= '9')) {
      int pos;
      if ('e' == ch) {
	cout << "Note that the vector starts with element 1.  Enter 0 if "
	     << endl
	     << "you have changed your mind and do not want to edit "<<endl
	     << "anything."<<endl;
	StreamUtil::promptRead
	  ("Which element do you want to edit? ", pos,
	   0, m_data->m_vec.size()+1);
      } else {
	pos = line.toInt ();
      }
      if (pos >= m_data->m_vec.size()+1) {
	cout << "Please enter an integer between 0 and "
	     << m_data->m_vec.size() << endl;
      } else if (0 < pos) {
	pos--;
	SP<SlotValue> oldValue = m_data->m_vec[pos]->copy();
	m_data->m_vec[pos]->editSlotValue();
	assert (oldValue->hasTheSameTypeAs (m_data->m_vec[pos]));
      }
    } else if (!m_data->m_constant && 'l' == ch) {
      if (0 == m_data->m_vec.size()) {
	cout << "No elements to delete!" << endl;
      } else {
	int pos;
	cout << "Enter 0 if you have changed your mind and no longer "
	     << endl
	     << "want to delete an element." << endl;
	StreamUtil::promptRead ("Which element do you want to delete? ", pos,
				0, m_data->m_vec.size()+1);
	if (0 < pos) {
	  pos--;
	  for (int i = pos; i < m_data->m_vec.size() - 1; i++) {
	    m_data->m_vec[i] = m_data->m_vec[i+1];
	  }
	  m_data->m_vec.pop();
	}
      }
    } else if ('p' == ch) {
      // Do nothing
    } else {
      cout << "Invalid command \"" << ch << "\".  Try again." << endl;
    }
  }
}

SP<SlotValue> VectorSlotValue::copy () const {
  return NEW (VectorSlotValue (*this));
}

int VectorSlotValue::size() const {
  return m_data->m_vec.size();
}

const SP<SlotValue> &VectorSlotValue::operator[] (int where) {
  return m_data->m_vec[where];
};

// They can use this to read from the vector, but not to write to it.
const SP<const SlotValue> VectorSlotValue::operator[] (int where) const {
  return &*m_data->m_vec[where]; 
};

void VectorSlotValue::setValue (int where, SP<SlotValue> sv) {
  assert (sv->hasTheSameTypeAs (m_data->m_default));
  m_data->m_vec[where] = sv;
}

SP<SlotValue> VectorSlotValue::getDefault () const {
  return m_data->m_default;
}

void VectorSlotValue::extendTo (int size) {
  int oldSize = m_data->m_vec.size();
  m_data->m_vec.extendTo (size);
  for (int i = oldSize; i < size; i++) {
    m_data->m_vec[i] = m_data->m_default->copy();
  }
}

const VectorSlotValue *VectorSlotValue::getSlotValue (const Configuration *a_c,
						      const String &name) {
  SP<const Configuration> c = a_c;
  SP<const SlotValue> slot = c->getSlot (name);
  const VectorSlotValue *vslot =
    dynamic_cast <const VectorSlotValue *> (&*slot);
  assert (vslot);
  return vslot;
}

VectorSlotValue *VectorSlotValue::getSlotValue (Configuration *a_c,
						const String &name)
{
  SP<Configuration> c = a_c;
  SP<SlotValue> slot = c->getSlot (name);
  VectorSlotValue *vslot =
    dynamic_cast <VectorSlotValue *> (&*slot);
  assert (vslot);
  return vslot;
}

void VectorSlotValue::setSlotValue (SP<Configuration> c,
				    const String &name,
				    SP<VectorSlotValue> rsv)
{
  SP<BadSlotValue> b = c->setSlot (name, &*rsv);
  assert (!b);
}

