// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __TypedRecursiveSlotValue_h__
#define __TypedRecursiveSlotValue_h__

#ifndef __RecursiveSlotValue_h__
#include "RecursiveSlotValue.h"
#endif

#ifndef __TypedFactory_h__
#include "TypedFactory.h"
#endif

#ifndef __TypedVersionOfFactory_h__
#include "TypedVersionOfFactory.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __Configuration_h__
#include "Configuration.h"
#endif

#ifndef __BadSlotValue_h__
#include "BadSlotValue.h"
#endif

template <class ConfClass, class ResultClass> class TypedRecursiveSlotValue
  : public RecursiveSlotValue
{
public:
  const TypedFactory <ConfClass, ResultClass> *getTypedFactory () const {
    return TypedVersionOfFactory <ConfClass, ResultClass>
      (RecursiveSlotValue::getFactory ());
  }
  SP<const ConfClass> getTypedConfiguration () const {
    SP<Configuration> c1 = RecursiveSlotValue::getConfiguration ();
    assert (c1);
    SP<const ConfClass> c2 = dynamic_cast <const ConfClass *> (&*c1);
    assert (c2);
    return c2;
  }
  SP<ConfClass> getTypedConfiguration () {
    SP<Configuration> c1 = RecursiveSlotValue::getConfiguration ();
    assert (c1);
    SP<ConfClass> c2 = dynamic_cast <ConfClass *> (&*c1);
    assert (c2);
    return c2;
  }
  // Next one allows setConfiguration applied to an SP<ConfClass>, but
  // unfortunately it also allows setConfiguration applied to any pointer to a
  // subclass of Configuration, since those calls will end up in 
  // RecursiveSlotValue's setConfiguration.
  void setConfiguration (ConfClass *c) {
    RecursiveSlotValue::setConfiguration (c);
  }
  // Like the above, except the compiler will complain about
  // setTypedConfiguration applied to a pointer to some irrelevant
  // configuration type.  It's better to use this than setConfiguration.
  void setTypedConfiguration (ConfClass *c) {
    RecursiveSlotValue::setConfiguration (c);
  }
  // This just blithely assumes that the given factory takes the right
  // configuration and returns the right type.
  TypedRecursiveSlotValue (const String &protocol, const Factory *f)
    : RecursiveSlotValue (protocol, f)
  {}
  // This just blithely assumes that the factory with the given name takes the
  // right configuration and returns the right type.
  TypedRecursiveSlotValue (const String &protocol, const String &factory)
    : RecursiveSlotValue (protocol, factory)
  {}
  SP<ResultClass> typedMakeIt () const {
    SP<Configurable> c1 = makeIt ();
    SP<ResultClass> c2;
    if (c1) {
      c2 = dynamic_cast <ResultClass *> (&*c1);
      assert (c2);
    }
    return c2;
  }
  // So far as typing goes, I think we want RecursiveSlotValue and
  // TypedRecursiveSlotValue to be identical.  This way we can have a slot
  // containing a RecursiveSlotValue, change it to a TypedRecursiveSlotValue,
  // and then change it back without getting errors.  If I implemented
  // hasTheSameTypeAs here to make TypedRecursiveSlotValue a separate type,
  // then you'd get an error when you replaced a TypedRecursiveSlotValue with a
  // RecursiveSlotValue.
  // FIXME Get users of hasTheSameTypeAs to accept some change in type.
  // They need to understand that changing to
  // a subtype and then changing back up to the original supertype is okay.
  // Nontrivial.  If we did it we could have RSV and
  // TRSV be distinct types.  The scheme used by PhysicsObjects is better than
  // the scheme used here, although I don't know if the staticClassId concept
  // would work with templates.  Looking at the standard seems to imply it
  // should, but I'd have to do some experiments before I really believe it.
  // Hmm, I think that it could be done, but we couldn't reliably initialize
  // everything at constructor time.  Thus staticClassId for a template class
  // would have to be a static function, and it would have to do work the first
  // time it was called.
  SP<SlotValue> copy () const {
    // Default copy constructor is sufficient.
    return NEW (TypedRecursiveSlotValue (*this));
  }
  static SP<TypedRecursiveSlotValue> getSlotValue
  (Configuration *c, const String &name)
  {
    SP<SlotValue> slot = c->getSlot (name);
    TypedRecursiveSlotValue *islot =
      dynamic_cast <TypedRecursiveSlotValue *> (&*slot);
    assert (islot);
    return islot;
  }
  static SP<const TypedRecursiveSlotValue> getSlotValue
  (const Configuration *c, const String &name)
  {
    SP<const SlotValue> slot = c->getSlot (name);
    const TypedRecursiveSlotValue *islot =
      dynamic_cast <const TypedRecursiveSlotValue *> (&*slot);
    assert (islot);
    return islot;
  }
  static void TypedRecursiveSlotValue::setSlotValue
  (Configuration *c, const String &name, TypedRecursiveSlotValue *rsv)
  {
    SP<BadSlotValue> b = c->setSlot (name, rsv);
    assert (!b);
  }
};
#endif
