// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __RecursiveSlotValue_h__
#define __RecursiveSlotValue_h__

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __SlotValue_h__
#include "SlotValue.h"
#endif

#ifndef __Factory_h__
#include "Factory.h"
#endif

#ifndef __Configuration_h__
#include "Configuration.h"
#endif

class Configurable;

// A slot that contains a Configurable that implements a particular
// protocol.  This kind of slot is special, since it has all the
// information we need to create something.
//
// FIXME: Restore old configuration upon restoring old factory
// Less cryptically, that is:
// If you set the factory, edit the configuration, set the factory to
// something else, and then set the factory back to the original
// value, it is more friendly to give back the old configuration, thus keeping
// those old edits, instead of generating a new configuration.
class RecursiveSlotValue
  : public SlotValue
{

  struct RecursiveSlotValueData;
  RecursiveSlotValueData *m_data;

  friend class ConstantRecursiveSlotValue;
  RecursiveSlotValue (const String &protocol, Factory *f, bool constantp);

public:

  RecursiveSlotValue (const RecursiveSlotValue &r);
  ~RecursiveSlotValue ();

  const Factory *getFactory () const;

  const Configuration *getConfiguration () const;
  Configuration *getConfiguration ();
  void setConfiguration (SP<Configuration> c);

  String getProtocol () const;
  // Can't set the protocol, it's constant forever.

  RecursiveSlotValue (const String &protocol, const Factory *f);

  // Look the factory and protocol up in the FactoryTable to get a
  // factory, and then use that as the factory.
  RecursiveSlotValue (const String &protocol, const String &factory);

  // I considered permitting starting with just a protocol, as in
  //    RecursiveSlotValue (const String &protocol);
  // but in that case a factory can easily have a default
  // configuration that isn't good for making things, and if someone
  // tries to make something we then have a complex UI problem of how
  // to lead them to the null sub-factory that they have to change to
  // actually do the creation.  Instead, we disallow null factories
  // here, so the calling factory has to put some factory here, and
  // the default configuration of a factory actually has to work to
  // run the factory.

  SP<Configurable> makeIt () const;
  bool hasTheSameTypeAs (const SlotValue *s) const;
  String unParse () const;
  void editSlotValue();
  SP<SlotValue> copy () const;
  // Don't want smart pointers as the arguments to getSlotValue here,
  // because otherwise every call that passes in a simple pointer is
  // ambiguous, since it doesn't know whether to take the const
  // version of getSlotValue or the non-const version.
  static SP<RecursiveSlotValue> getSlotValue (Configuration * c,
					      const String &name);
  static SP<const RecursiveSlotValue> getSlotValue (const Configuration * a_c,
						    const String &name);
  static void setSlotValue (SP<Configuration> c,
			    const String &name,
			    SP<RecursiveSlotValue> rsv);
};

#endif
