// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#include "RecursiveSlotValue.h"

#ifndef __Configurable_h__
#include "Configurable.h"
#endif

#ifndef __Factory_h__
#include "Factory.h"
#endif

#ifndef __TextConfigurationEditor_h__
#include "TextConfigurationEditor.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __StreamUtil_h__
#include "StreamUtil.h"
#endif

#ifndef __iostream_h__
#include <iostream.h>
#define __iostream_h__
#endif

#ifndef __VecUtil_h__
#include "VecUtil.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

// For abort.
#ifndef __stdlib_h__
#include <stdlib.h>
#define __stdlib_h__
#endif

#ifndef __BadSlotValue_h__
#include "BadSlotValue.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

struct RecursiveSlotValue::RecursiveSlotValueData {
  // The protocol that this RecursiveSlotValue implements.  Cannot
  // simply be inferred from the factory, now that we have a subclass
  // relationship among protocols.
  String m_protocol;

  // The factory that will compute it.  Cannot be null.  Cannot change.
  // If you want to change the Factory, then instead call setSlot on
  // the Configuration that this RecursiveSlotValue is part of.
  SP<const Factory> m_factory;

  // The configuration for the factory.  The default configuration for
  // the factory is good here.  This configuration can be edited in
  // place.  
  // 
  // Another choice would have been to prevent modifying the
  // configuration in place.  In this case only the top-level
  // configuration would be editable in place, and all recursive
  // configurations would be frozen, requiring rebuilding the
  // configuration and calling setSlotValue higher up to modify the
  // configuration.  It's silly to have some configurations editable
  // and not others, so actually we would want the top-level
  // configuration to not be editable either, and all you can do is
  // build a new top-level configuration and then set the top-level
  // configuration to a new value.  But we want to allow the UI to
  // update in place when a configuration is changed, and detecting
  // such a change will be complex if the only thing you can do is
  // changing by copying and replacing.  Also, the complexity of
  // rebuilding the configurations from the bottom whenever anything
  // changes seems like a cost without much benefit.
  SP<Configuration> m_configuration;
  // Whether the factory is constant.  The configuration is always editable.
  bool m_constant;
  RecursiveSlotValueData () {
    m_constant = false;
  }
};

static SP<Factory> safeGetFactory (const String &protocol,
				   const String &factory) {
  SP<Factory> f = FactoryTable::load (protocol, factory);
  if (f) {
    return f;
  } else {
    cerr << "Could not load factory " << factory << " in protocol "
	 << protocol << endl;
    abort ();
  }
}

RecursiveSlotValue::RecursiveSlotValue (const RecursiveSlotValue &r)
  : SlotValue (r),
    m_data (NEW (RecursiveSlotValueData ()))
{
  m_data->m_protocol = r.getProtocol ();
  m_data->m_factory = r.getFactory();
  m_data->m_configuration = r.getConfiguration()->copy();
  m_data->m_constant = r.m_data->m_constant;
}

RecursiveSlotValue::RecursiveSlotValue (const String &protocol,
					const String &factory)
  : SlotValue (protocol),
    m_data (NEW (RecursiveSlotValueData ()))
{
  m_data->m_protocol = protocol;
  m_data->m_factory = safeGetFactory (protocol, factory);
  m_data->m_configuration =
    safeGetFactory (protocol, factory)-> defaultConfiguration();
  m_data->m_constant = false;
}

// Don't have SP<Factory> here, since every call to the constructor
// will then be ambiguous.  Factory* converts to bool converts to
// String converts to const String &.
RecursiveSlotValue::RecursiveSlotValue (const String &protocol,
					const Factory *factory)
  : SlotValue (protocol),
    m_data (NEW (RecursiveSlotValueData ()))
{
  m_data->m_protocol = protocol;
  m_data->m_factory = factory;
  m_data->m_configuration = factory->defaultConfiguration();
  m_data->m_constant = false;
}

// Don't have SP<Factory> here, since every call to the constructor
// will then be ambiguous.  Factory* converts to bool converts to
// String converts to const String &.
RecursiveSlotValue::RecursiveSlotValue (const String &protocol,
					Factory *factory,
					bool constant)
  : SlotValue (protocol),
    m_data (NEW (RecursiveSlotValueData ()))
{
  m_data->m_protocol = protocol;
  m_data->m_factory = factory;
  m_data->m_configuration = factory->defaultConfiguration();
  assert (constant);
  m_data->m_constant = constant;
}

RecursiveSlotValue::~RecursiveSlotValue () {
  assert (m_data);
  delete m_data;
  m_data = 0;
}

String RecursiveSlotValue::getProtocol () const {
  return m_data->m_protocol;
}

const Factory *RecursiveSlotValue::getFactory () const {
  assert (m_data->m_factory);
  return m_data->m_factory;
}

const Configuration *RecursiveSlotValue::getConfiguration () const {
  assert (m_data->m_configuration);
  return m_data->m_configuration;
}

Configuration *RecursiveSlotValue::getConfiguration () {
  assert (m_data->m_configuration);
  return m_data->m_configuration;
}

void RecursiveSlotValue::setConfiguration (SP<Configuration> c) {
  assert (c->hasTheSameTypeAs (&*m_data->m_configuration));
  m_data->m_configuration = c;
}

SP<Configurable> RecursiveSlotValue::makeIt () const {
  return m_data->m_factory->makeIt (&*m_data->m_configuration);
}

bool RecursiveSlotValue::hasTheSameTypeAs (const SlotValue *s) const {
  SP<const RecursiveSlotValue> theRSV =
    dynamic_cast <const RecursiveSlotValue *> (s);
  bool result = (theRSV && getProtocol() == theRSV->getProtocol());
  return result;
}

String RecursiveSlotValue::unParse () const {
  const String factoryName =
    m_data->m_factory->getPrettyName (m_data->m_configuration);
  const String confUnParse = m_data->m_configuration->unParse ();
  if ("" == confUnParse) {
    return factoryName;
  } else {
    return factoryName + "//" + confUnParse;
  }
}

bool cmpFactory (const SP<Factory> &f1, const SP<Factory> &f2) {
  return f1->getName() < f2->getName();
}

void RecursiveSlotValue::editSlotValue () {
  cout << "This slot implements the protocol "<<
    getProtocol()<<"."<<endl;
  cout << "The currently chosen factory is "<<
    getFactory()->getName()<<"."<<endl;
  bool changeFactory;
  if (m_data->m_constant) {
    changeFactory = false;
  } else {
    StreamUtil::promptRead
      ("Do you want to change the factory and reset to the new factory's\n"
       "default configuration? ",
       changeFactory);
  }
  if (changeFactory) {
    Dynavec <SP<Factory> > factories =
      FactoryTable::load (getProtocol());
    VecUtil::sort (factories, cmpFactory);
    if (0 == factories.size ()) {
      cout << "There are no factories for the protocol "
	   << getProtocol()<<"."<<endl
	   << "This is a bug, probably a misspelled protocol name somewhere."
	   << endl;
      assert (factories.size ());
      cout << "Leaving the current factory unchanged."<<endl;
    } else {
      cout << "The available factories are:" << endl;
      Dynavec <Dynavec <String> > tableau;
      tableau.push (Dynavec <String> ());
      tableau[0].push ("Number");
      tableau[0].push ("Name");
      for (int i = 0; i < factories.size(); i++) {
	tableau.push (Dynavec <String> ());
	tableau[tableau.size()-1].push (i);
	tableau[tableau.size()-1].push (factories[i]->getName());
      }
      StreamUtil::printTableau (cout, tableau);
      int pos;
      StreamUtil::promptRead ("What is the number of the factory you want? ",
			      pos, 0, factories.size());
      SP<Factory> newFactory = factories [pos];
      m_data->m_factory = newFactory;
      m_data->m_configuration = newFactory->defaultConfiguration();
    }
  }
  cout << "Editing the configuration..." << endl;
  TextConfigurationEditor::edit (getConfiguration());
}

SP<SlotValue> RecursiveSlotValue::copy () const {
  return NEW (RecursiveSlotValue (*this));
}

SP<RecursiveSlotValue> RecursiveSlotValue::getSlotValue
(Configuration * a_c, const String &name)
{
  SP<Configuration> c = a_c;
  SP<SlotValue> slot = c->getSlot (name);
  RecursiveSlotValue *islot =
    dynamic_cast <RecursiveSlotValue *> (&*slot);
  assert (islot);
  return islot;
}

SP<const RecursiveSlotValue> RecursiveSlotValue::getSlotValue
(const Configuration *a_c, const String &name)
{
  SP<const Configuration> c = a_c;
  SP<const SlotValue> slot = c->getSlot (name);
  const RecursiveSlotValue *islot =
    dynamic_cast <const RecursiveSlotValue *> (&*slot);
  assert (islot);
  return islot;
}

void RecursiveSlotValue::setSlotValue (SP<Configuration> c,
				       const String &name,
				       SP<RecursiveSlotValue> rsv)
{
  SP<BadSlotValue> b = c->setSlot (name, &*rsv);
  assert (!b);
}

#ifdef UNIT_TEST

// Compilation command is
// g++ -Wall -Wwrite-strings -Werror -pipe -felide-constructors -g  -I/home/tim/fungimol/Util -I/home/tim/fungimol/Hashtable -I/home/tim/fungimol/String -DUNIT_TEST -o foo /home/tim/fungimol/Configuration/RecursiveSlotValue.cpp Configuration.a ../String/String.a ../Hashtable/Hashtable.a ../Util/Util.a

#ifndef __Configurable_h__
#include "Configurable.h"
#endif

class BoolProtocol : public Configurable {
public:
  virtual bool getValue () = 0;
  virtual ~BoolProtocol () {};
};

#ifndef __Factory_h__
#include "Factory.h"
#endif

#ifndef __IntSlotValue_h__
#include "IntSlotValue.h"
#endif

class BoolFactory2Configuration : public Configuration {
public:
  BoolFactory2Configuration () {
    initializeSlot ("value", NEW (IntSlotValue (0)));
  }
};

class Bool2 : public BoolProtocol {
  const bool m_value;
public:
  Bool2 (bool value) : m_value (value) {};
  bool getValue () {
    return m_value;
  }
};

class BoolFactory2 : public Factory {
public:
  BoolFactory2 () : Factory ("BoolProtocol", "BoolFactory2") {}
  BoolFactory2Configuration *defaultConfiguration () const {
    return NEW (BoolFactory2Configuration ());
  }
  Bool2 *makeIt (SP<Configuration> c) const {
    SP<IntSlotValue> slot =
      dynamic_cast <IntSlotValue *> (c->getSlot ("value"));
    assert (slot);
    bool value = 0 != (slot->getValue());
    return NEW (Bool2 (value));
  }
};

const static bool useless2 = (FactoryTable::store (NEW (BoolFactory2 ())),
			      true);

class BoolFactory1Configuration : public Configuration {
public:
  BoolFactory1Configuration () {
    initializeSlot ("copyFrom",
		    NEW (RecursiveSlotValue
			 (FactoryTable::load ("BoolProtocol",
					      "BoolFactory2"))));
  }
};

class Bool1 : public BoolProtocol {
  SP<BoolProtocol> m_other;
public:
  Bool1 (SP<BoolProtocol> theOther)
    : m_other (theOther) {
    assert (theOther);
  }
  bool getValue () {
    return m_other->getValue();
  }
};

class BoolFactory1 : public Factory {
public:
  BoolFactory1 () : Factory ("BoolProtocol", "BoolFactory1") {}
  Configuration *defaultConfiguration () const {
    return NEW (BoolFactory1Configuration ());
  }
  // The following function has enough casts for ten.
  Bool1 *makeIt (SP<Configuration> c) const {
    assert (dynamic_cast <const BoolFactory1Configuration *>
	    ((const Configuration *)c));
    SP<const BoolFactory1Configuration> conf =
      (const BoolFactory1Configuration *) ((const Configuration *) c);
    return
      NEW (Bool1 (dynamic_cast <BoolProtocol *>
		  ((Configurable *)
		   ((RecursiveSlotValue *)
		    (c->getSlot ("copyFrom")))->makeIt ())));
  }
};

const static bool useless1 = (FactoryTable::store (NEW (BoolFactory1 ())),
			      true);

class BoolFactorynotConfiguration : public Configuration {
public:
  BoolFactorynotConfiguration () {
    initializeSlot ("copyFrom",
		    NEW (RecursiveSlotValue
			 (FactoryTable::load ("BoolProtocol",
					      "BoolFactory2"))));
  }
};

class Boolnot : public BoolProtocol {
  SP<BoolProtocol> m_other;
public:
  Boolnot (SP<BoolProtocol> theOther) : m_other (theOther) {};
  bool getValue () {
    return !(m_other->getValue());
  }
};

class BoolFactorynot : public Factory {
public:
  BoolFactorynot () : Factory ("BoolProtocol", "BoolFactorynot") {}
  Configuration *defaultConfiguration () const {
    return NEW (BoolFactorynotConfiguration ());
  }
  BoolProtocol *makeIt (SP<Configuration> c) const {
    SP<RecursiveSlotValue> slot =
      dynamic_cast <RecursiveSlotValue *>(c->getSlot ("copyFrom"));
    assert (slot);
    SP<BoolProtocol> v =
      dynamic_cast <BoolProtocol *>((Configurable *)(slot->makeIt()));
    return
      NEW (Boolnot (v));
  }
};

const static bool useless = (FactoryTable::store (NEW (BoolFactorynot ())),
			     true);

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __TextConfigurationEditor_h__
#include "TextConfigurationEditor.h"
#endif

int main (int argc, char **argv) {
  SP<Factory> f = FactoryTable::load ("BoolProtocol", "BoolFactory1");
  SP<Configuration> c = f->defaultConfiguration();
  for (;;) {
    c = TextConfigurationEditor::edit (c);
    cout << "Result is "<< ((BoolProtocol *)f->makeIt(c))->getValue()<< endl;
  }
}

#endif
