// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//
#ifndef __MenuSlotValueChoices_h__
#include "MenuSlotValueChoices.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __VecUtil_h__
#include "VecUtil.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

struct MenuSlotValueChoices::MenuSlotValueChoicesData {
  Dynavec <String> m_short;
  Dynavec <String> m_pretty;
  int m_generation;
  inline void check () const {
    assert (m_short.size () == m_pretty.size ());
  }
};

void MenuSlotValueChoices::addChoice (const String &shortName, const String &prettyName)
{
  m_data->check ();
  m_data->m_generation ++;
  // Only add it if the shortName is new.  If the shortName is old and the
  // prettyName is new, then we have a silent, slightly bad outcome.  Not worth
  // adding the machinery for coping with this.
  if (-1 == lookUpShortName (shortName)) {
    m_data->m_short.push (shortName);
    m_data->m_pretty.push (prettyName);
  }
}

const String &MenuSlotValueChoices::getPrettyName (int pos) const {
  m_data->check ();
  return m_data->m_pretty [pos];
}

const String &MenuSlotValueChoices::getShortName (int pos) const {
  m_data->check ();
  return m_data->m_short [pos];
}

int MenuSlotValueChoices::size () const {
  m_data->check ();
  return m_data->m_short.size ();
}

int MenuSlotValueChoices::lookUpShortName (const String &shortName) const {
  int result = 0;
  bool found = VecUtil::find (shortName, m_data->m_short, &result);
  if (!found) {
    result = -1;
  }    
  return result;
}

MenuSlotValueChoices::MenuSlotValueChoices ()
  : m_data (NEW (MenuSlotValueChoicesData ()))
{}

MenuSlotValueChoices::~MenuSlotValueChoices () {
  assert (m_data);
  delete m_data;
  m_data = 0;
}

int MenuSlotValueChoices::generation () const {
  m_data->check ();
  return m_data->m_generation;
}
