// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//
#ifndef __MenuSlotValue_h__
#include "MenuSlotValue.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __MenuSlotValueChoices_h__
#include "MenuSlotValueChoices.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __iostream_h__
#include <iostream.h>
#define __iostream_h__
#endif

#ifndef __StreamUtil_h__
#include "StreamUtil.h"
#endif

#ifndef __Configuration_h__
#include "Configuration.h"
#endif

#ifndef __SlotValue_h__
#include "SlotValue.h"
#endif

struct MenuSlotValue::MenuSlotValueData
{
  // The current menu choice.
  // The default copy constructor is fine for this.
  int m_index;
  SP<MenuSlotValueChoices> m_choices;
  inline void check () const {
    assert (m_choices->size());
    assert (0 <= m_index);
    assert (m_index < m_choices->size());
  }
};

int MenuSlotValue::getIntChoice () const {
  m_data->check ();
  return m_data->m_index;
}

String MenuSlotValue::getShortStringChoice () const {
  m_data->check ();
  return m_data->m_choices->getShortName (m_data->m_index);
}

String MenuSlotValue::getPrettyStringChoice () const {
  m_data->check ();
  return m_data->m_choices->getPrettyName (m_data->m_index);
}

void MenuSlotValue::setIntChoice (int newValue) {
  m_data->m_index = newValue;
  m_data->check ();
}

void MenuSlotValue::setShortStringChoice (const String &choice) {
  setIntChoice (m_data->m_choices->lookUpShortName (choice));
}

SP<MenuSlotValueChoices> MenuSlotValue::getChoices () {
  return m_data->m_choices;
}

SP<const MenuSlotValueChoices> MenuSlotValue::getChoices () const {
  return &*(m_data->m_choices);
}

MenuSlotValue::MenuSlotValue (MenuSlotValueChoices *choices)
  : SlotValue ("Menu"),
    m_data (NEW (MenuSlotValueData ()))
{
  m_data->m_index = 0;
  m_data->m_choices = choices;
  // Don't call check so that the initial list of choices can be empty.
}

MenuSlotValue::~MenuSlotValue () {
  assert (m_data);
  delete m_data;
  m_data = 0;
}

void MenuSlotValue::editSlotValue ()
{
  cout << "The choices are:" << endl;
  for (int i = 0; i < m_data->m_choices->size(); i++) {
    cout << i << ":  " << m_data->m_choices->getPrettyName (i) << endl;
  }
  StreamUtil::promptRead ("Which choice do you want? ",
			  m_data->m_index, 0, m_data->m_choices->size());
}

bool MenuSlotValue::hasTheSameTypeAs (const SlotValue *s) const {
  return 0 != dynamic_cast <const MenuSlotValue *> (s);
}

MenuSlotValue::MenuSlotValue (const MenuSlotValue &msv)
  : SlotValue (msv),
    m_data (NEW (MenuSlotValueData (*(msv.m_data))))
{}

SP<SlotValue> MenuSlotValue::copy () const {
  return NEW (MenuSlotValue (*this));
}

String MenuSlotValue::unParse () const {
  return m_data->m_choices->getPrettyName (m_data->m_index);
}

SP<MenuSlotValue> MenuSlotValue::getSlotValue (Configuration *a_c,
					       const String &name) {
  SP<Configuration> c = a_c;
  SP<SlotValue> slot = c->getSlot (name);
  assert (slot);
  MenuSlotValue *result = dynamic_cast <MenuSlotValue *> (&*slot);
  assert (result);
  return result;
}

SP<const MenuSlotValue> MenuSlotValue::getSlotValue (const Configuration *a_c,
						     const String &name) {
  SP<const Configuration> c = a_c;
  SP<const SlotValue> slot = c->getSlot (name);
  assert (slot);
  const MenuSlotValue *result = dynamic_cast <const MenuSlotValue *> (&*slot);
  assert (result);
  return result;
}

