// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#include "FactoryTable.h"

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __HashTable_h__
#include "HashTable.h"
#endif

#ifndef __Refcount_h__
#include "Refcount.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __Factory_h__
#include "Factory.h"
#endif

#ifndef __VecUtil_h__
#include "VecUtil.h"
#endif

struct FactoryList : public Refcount {
  Dynavec <SP<Factory> > m_vec;
};

// Instead of a static member in the FactoryTable class, we have a
// file-scope member in this file.  This way, if we decide to change
// the representation, people who use FactoryTable will not have to be
// recompiled.
// store and load have to work during static initialization.  Page 19
// of ARM says that statics are initialized to zero, then run-time
// initialization happens, so we'll rely upon table being initialized
// to zero and initialize it on the fly in load if it's still zero.
// But initializing it to zero after we've loaded everybody else is
// bad too.  Use the following hack.  Don't use smart pointers to
// initialize the table, since in this case the weird recursion here
// causes it to be left with a positive reference count.
static HashTable <String, FactoryList> * initFTable ();

static HashTable <String, FactoryList> * factoryTable = initFTable ();

static HashTable <String, FactoryList> * initFTable () {
  if (factoryTable) {
    return factoryTable;
  } else {
    factoryTable = NEW2 (HashTable <String, FactoryList> ());
    factoryTable->ref();
    return factoryTable;
  }
}

struct SubclassList : public Refcount {
  Dynavec <String> m_vec;
};

static HashTable <String, SubclassList> * initSTable ();

static HashTable <String, SubclassList> * subclassTable = initSTable ();

static HashTable <String, SubclassList> * initSTable () {
  if (subclassTable) {
    return subclassTable;
  } else {
    subclassTable = NEW2 (HashTable <String, SubclassList> ());
    subclassTable->ref();
    return subclassTable;
  }
}

static void initTables () {
  initFTable ();
  initSTable ();
}

void FactoryTable::store (const String &protocol, SP <Factory> f) {
  initTables();
  assert (f);
  SP<FactoryList> fl = factoryTable->load (protocol);
  if (! fl) {
    fl = NEW (FactoryList ());
    factoryTable->store (protocol, fl);
  }
  if (! VecUtil::find (f, fl->m_vec)) {
    fl->m_vec.push (f);
  }
}

static void pushProtocolFactories (const String &protocol,
				   Dynavec <SP<Factory> > &result)
{
  SP<FactoryList> fl = factoryTable->load (protocol);
  if (fl) {
    for (int i = 0; i < fl->m_vec.size(); i++) {
      if (!VecUtil::find (fl->m_vec[i], result)) {
	result.push (fl->m_vec[i]);
      }
    }
  }
  SP<SubclassList> cl = subclassTable->load (protocol);
  if (cl) {
    for (int i = 0; i < cl->m_vec.size (); i++) {
      pushProtocolFactories (cl->m_vec[i], result);
    }
  }
}

Dynavec <SP<Factory> > FactoryTable::load (const String &protocol) {
  initTables ();
  Dynavec <SP<Factory> > result;
  pushProtocolFactories (protocol, result);
  return result;
}

SP<Factory> FactoryTable::load (const String &protocol,
				const String &factoryName)
{
  initTables ();
  SP<Factory> result;
  Dynavec <SP<Factory> > fl = load (protocol);
  if (fl.size()) {
    for (int i = 0; i < fl.size(); i++) {
      if (factoryName == fl[i]->getName()) {
	result = fl[i];
      }
    }
  }
  return result;
}

void FactoryTable::registerSubclass (const String &subclass,
				     const String &superclass) {
  SP<SubclassList> cl = subclassTable->load (superclass);
  if (! cl) {
    cl = NEW (SubclassList ());
    subclassTable->store (superclass, cl);
  }
  if (!VecUtil::find (subclass, cl->m_vec)) {
    cl->m_vec.push (subclass);
  }
}

namespace {
  void deAllocate () {
    factoryTable->deref();
    assert (0 == factoryTable->count ());
    delete (factoryTable);
    factoryTable = 0;
    subclassTable->deref();
    assert (0 == subclassTable->count ());
    delete (subclassTable);
    subclassTable = 0;
  }
  static bool useless = (MemoryUtil::registerDeallocator (deAllocate),
			 true);
}
