// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __Configuration_h__
#define __Configuration_h__

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __Refcount_h__
#include "Refcount.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

class String;
class SlotValue;
class BadSlotValue;

// A specification for making a thing.
// You make one of these, then you call methods on it to fill in the
// fields.  Setting the fields gives it a default value.  Normally you
// make a subclass that has the methods you want; this way it's
// type-safe.  A UI should call getSlots, getSlot, and setSlot to set
// the values.
// Note that not all the state information in the Configuration has to be
// published so the UI can manupulate it.  If the state information is
// there, but there is no slot for it, then it is invisible to the UI.

// Slots can also be modified in place.  The procedure used in
// TextConfigurationEditor is: make a copy of the old slot, modify it in place,
// call setSlot to set it to its current value, and if setSlot indicates an
// error by returning a non-null BadSlotValue, the value is set back to the
// safe copy.  This allows slots to edit themselves in place, but still setSlot
// can be overridden to constrain what values can be placed into a slot.

class Configuration : public Refcount {
 private:
  class ConfigurationData;
  // The real instance data is hidden behind this pointer, so
  // implementation changes that aren't interface changes won't
  // require the users to recompile.
  ConfigurationData * m_data;

  // Unimplemented; we can't assign Configurations because in general
  // there is other stuff carried along with them, and this class
  // isn't given knowledge about what the other stuff is.
  Configuration &operator= (const Configuration &c);
 protected:
  // Subclasses can copy so they can implement the copy method.
  Configuration (const Configuration &c);
  // Subclasses of this can set up initial values for slots.  The
  // following method will abort if the slot that would be initialized already
  // has a value.
  void initializeSlot (const String &name, SlotValue *sv);
 public:
  // Return the names of the slots, in random order.
  virtual Dynavec <String> getSlots () const;
  // Return the current value of one slot, or null if the name of the
  // slot is not recognized.
  virtual SP<SlotValue> getSlot (const String &name);
  virtual SP<const SlotValue> getSlot (const String &name) const;

  // Set a value for one slot.  The new value should have the same
  // type as the old value.  The SlotValue won't be copied.  
  // Virtual to allow alternative representations of slots in
  // subclasses of Configuration.
  // If setting the slot succeeded, return null; if it failed, return
  // a BadSlotValue object that says why.  This would be an exception,
  // if I were willing to slow down this entire program by compiling
  // with exception handling turned on.
  virtual SP<BadSlotValue> setSlot (const String &name, SP<SlotValue> sv);
  // If the public makes a subclass of Configuration, then it simply
  // has no slots.
  Configuration ();
  virtual ~Configuration ();
  virtual SP<Configuration> copy () const = 0;
  virtual bool hasTheSameTypeAs (const Configuration *c) const = 0;
  // Default unparser returns the empty string.
  // The unparser will normally say what the type of the configuration is, such
  // as "CursorBallOrbMove Configuration".  We use this when there's a vector
  // of ConstantRecursiveSlotValues, which tries to unparse the configurations
  // when asked to print out the vector for the user to edit.
  virtual String unParse () const;
};

#endif
