// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#include "Configuration.h"

// Next one for abort.
#ifndef __stdlib_h__
#include <stdlib.h>
#define __stdlib_h__
#endif

#ifndef __HashTable_h__
#include "HashTable.h"
#endif

#ifndef __iostream_h__
#include <iostream.h>
#define __iostream_h__
#endif

#ifndef __SlotValue_h__
#include "SlotValue.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

namespace {
  // This is used only within this file, so I suppose it's okay to have
  // public fields. 
  struct SlotValueInfo : public Refcount {
    SP<SlotValue> m_sv;
    SlotValueInfo (SlotValue *sv) {
      m_sv = sv;
    }
  };
}

#ifndef __String_h__
#include "String.h"
#endif

// Likewise, this is file-scope so I suppose it's okay to have a
// public field.
class Configuration::ConfigurationData {
public:
  SP<HashTable <String, SlotValueInfo> > m_slots;
  ConfigurationData () : m_slots (NEW2 (HashTable <String, SlotValueInfo>))
  {}
};

Configuration::Configuration () : m_data (NEW (ConfigurationData ())) {};

Configuration::~Configuration () {
  assert (m_data);
  delete m_data;
  m_data = 0;
};

void Configuration::initializeSlot (const String &name, SlotValue *sv)
{
  assert (m_data);
  if (m_data->m_slots->load (name)) {
    cerr << "Attempt to initialize slot " << name <<
      " which is already initialized." << endl;
    abort();
  } else {
    m_data->m_slots->store (name, NEW (SlotValueInfo (sv)));
  }
}

#ifndef __Dynavec_h__
#include "Dyanvec.h"
#endif

Dynavec <String> Configuration::getSlots () const {
  Dynavec <String> result;
  assert (m_data);
  HashTable <String, SlotValueInfo>::HashTableIterator i =
    m_data->m_slots->iterator();
  for (;;) {
    if (i.isDone ()) break;
    result.push (i.currentItem());
    i.next();
  }
  return result;
}

SP<SlotValue> Configuration::getSlot (const String &name) 
{
  assert (m_data);
  SP<const SlotValueInfo> svi = m_data->m_slots->load (name);
  if (!svi) {
    cerr << "Attempt to get the slot "<< name
	 << " which does not exist." << endl;
    abort();
  };
  return svi->m_sv;
}

SP<const SlotValue> Configuration::getSlot (const String &name) const
{
  assert (m_data);
  SP<const SlotValueInfo> svi = &*(m_data->m_slots->load (name));
  if (!svi) {
    cerr << "Attempt to get the slot "<< name
	 << " which does not exist." << endl;
    abort();
  };
  return &*(svi->m_sv);
}

#ifndef __BadSlotValue_h__
#include "BadSlotValue.h"
#endif

SP<BadSlotValue> Configuration::setSlot (const String &name, SP<SlotValue> sv)
{
  assert (m_data);
  SP<SlotValueInfo> svi = m_data->m_slots->load (name);
  if (!svi) {
    cerr << "Attempt to set the slot "<<name
	 << " but the slot has not been initialized."<<endl;
    abort();
  }
  if (! (svi->m_sv->hasTheSameTypeAs(sv))) {
    cerr << "Attempt to change the type of the slot "<<name
	 <<"."<<endl;
    abort();
  }
  SP<BadSlotValue> b = sv->constrain ();
  if (b) {
    return b;
  } else {
    svi->m_sv = sv;
    return 0;
  }
}

Configuration::Configuration (const Configuration &c)
  : Refcount (c),
    m_data (NEW (ConfigurationData ()))
{
  HashTable<String, SlotValueInfo>::HashTableIterator i =
    c.m_data->m_slots->iterator();
  for (;;) {
    if (i.isDone ()) break;
    SP<SlotValueInfo> svi = c.m_data->m_slots->load (i.currentItem());
    assert (svi);
    m_data->m_slots->store (i.currentItem(),
			   NEW (SlotValueInfo (svi->m_sv->copy())));
    i.next();
  }
}

String Configuration::unParse () const {
  return "";
}
