// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

// For strerror.
#ifndef  __string_h__
#include <string.h>
#define __string_h__
#endif

// For errno.
#ifndef __errno_h__
#include <errno.h>
#define __errno_h__
#endif

#ifndef __fstream_h__
#include <fstream.h>
#define __fstream_h__
#endif

#ifndef __TypedFactory_h__
#include "TypedFactory.h"
#endif

#ifndef __Action_h__
#include "Action.h"
#endif

#ifndef __NullObject_h__
#include "NullObject.h"
#endif

#ifndef __SceneGraph_h__
#include "SceneGraph.h"
#endif

#ifndef __LinkManager_h__
#include "LinkManager.h"
#endif

#ifndef __AtomConfiguration_h__
#include "AtomConfiguration.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __AtomInfo_h__
#include "AtomInfo.h"
#endif

#ifndef __TopLevel_h__
#include "TopLevel.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __OL_h__
#include "OL.h"
#endif

#ifndef __BoringAtom_h__
#include "BoringAtom.h"
#endif

#ifndef __PlayBrennerMovieConfiguration_h__
#include "PlayBrennerMovieConfiguration.h"
#endif

class Canvas;
namespace {
  class MoviePlayer
    : public NullObject
  {
    mutable bool m_playing;
    // The index into the following vectors is the index of the link to the
    // object we're moving around.  The two vectors should have the same
    // length.
    //
    // The last observed positions of each object.  If someone else has moved
    // it since then, we have to update m_offsets.
    mutable Dynavec <Vec3> m_lastPosition;
    // The difference between the position of the object according to the movie
    // file we're reading and where we want to put the object in the scene
    // graph.  By keeping this around, we are able to use editing commands to
    // adjust where in the scene graph the movie plays back, which is clearly
    // useful.  We can also use editing commands to move some of the atoms in
    // the replaying movie somewhere else; this might conceivably be good for
    // something.  Unfortunately, we can also use editing commands to rotate
    // the playback of the movie without rotating the reference frame being
    // used for playback, thus causing an obviously useless display of
    // meaningless nonsense.
    //
    // FIXME Object rotation code should refuse to rotate objects in movies.
    //
    // Discovering which objects shouldn't be rotated as part of the selection
    // will require serious digging around in the scene graph to find the
    // movies.  Ugly.  Could cache it in the SelectionManager just like we
    // cache the invsibility information.
    //
    // Or we could augment the PhysicsObject protoocol to be able to say that
    // some objects aren't rotatable, and cause the animation objects in a
    // movie to mark themselves as nonrotatable, and the code for rotating the
    // selection checks for this.  Maybe we could just permit
    // PhysicsObject::rotate to return a bool indicating whether the rotation
    // is permitted.
    mutable Dynavec <Vec3> m_offsets;
    // Have to use an array of ObjectLink's here instead of using
    // getLinkManager()->getLinks, because we want to delete the objects from
    // our destructor.  See the voluminous comment before the virtual
    // destructor declaration in SceneGraph.h.
    mutable Dynavec <OL<BoringAtom> > m_links;
    inline void check () const {
      // Call this immediately after observing that this object has count
      // outgoing links to verify that the array lengths are consistent.
      assert (m_lastPosition.size () == m_links.size());
      assert (m_offsets.size () == m_links.size());
    }
    // The input file we're playing from.
    mutable ifstream stream;
    // Whether we've shown any frames yet.  For the first frame, we need to
    // create the objects that we'll move around during the animation.
    mutable bool m_firstFrame;
    // Error message from opening the stream, or the empty string if everything
    // is okay.  Errors while reading are different: we call "message" on them
    // and do nothing else with them.
    String m_error;
    // For error reporting on close.
    String m_fileName;
    // Factory to create the BoringAtom's we use for the animation.
    SP<Factory> m_boringAtomFactory;
    // Configuration for creating atoms.
    mutable SP<AtomConfiguration> m_AtomConfiguration;
    // Used for positioning new atoms.
    mutable Dynavec <Float> m_plausibleState;
    mutable ObjectLink m_myLink;
    const bool m_repeat;
    void shutDown () const {
      m_playing = false;
      if (stream.is_open ()) {
	stream.close ();
	// We can get bogus error messages on close, so don't check.
      }
      for (int i = 0; i < m_links.size(); i++) {
	if (m_links[i].isValid ()) {
	  m_links [i].deleteObject();
	}
      }
      if (m_myLink.isValid ()) {
	m_myLink.deleteObject ();
	// Not safe to do any computation after this point, because this may be
	// destructed.
      }
    }
    void initMovieState () const {
      for (int i = 0; i < m_links.size(); i++) {
	if (m_links[i].isValid ()) {
	  m_links[i].deleteObject ();
	}
      }
      m_playing = true;
      m_firstFrame = true;
      m_links.extendTo (0);
      m_lastPosition.extendTo (0);
      m_offsets.extendTo (0);
    }
  public:
    MoviePlayer (const String &fileName, bool repeat)
      : m_repeat (repeat)
    {
      m_fileName = fileName;
      stream.open (&*fileName);
      if (!stream.is_open ()) {
	m_error = (String ("Could not open ") + fileName + " for reading: "
		   + strerror (errno));
      }
      m_boringAtomFactory = FactoryTable::load ("Atom", "BoringAtom");
      m_AtomConfiguration = NEW (AtomConfiguration ());
      initMovieState ();
    }
    ~MoviePlayer () {
      // Will stream's destructor will close it for us?  Probably, but I'm not
      // sure.
      ref ();
      shutDown ();
      deref ();
    }

    void setPlaying (bool playing) const {
      m_playing = playing;
    }

    bool getPlaying () const {
      return m_playing;
    }

    String getError () const {
      return m_error;
    }

    bool hasAnimationEver () const {
      return true;
    }

    bool hasAnimationNow (const SceneGraph *sg, int index) const {
      (void) sg; (void) index;
      bool result;
      if (!m_playing) {
	result = false;
      } else {
	assert (stream.is_open());
	check ();
	result = false;
	for (int i = 0; i < m_links.size(); i++) {
	  if (m_links[i].isValid ()) {
	    result = true;
	    break;
	  }
	}
      }
      return result;
    }
			  
    void animateFrame (Canvas *canvas, SceneGraph *sg, int index) const {
      (void) canvas;
      m_myLink = ObjectLink (sg, index);
      check ();
      assert (stream.is_open ());
      String line;
      // shutDown may delete this object, so postpone all shutdowns to the end.
      bool needShutdown = false;
      if (m_playing) {
	// The header, which we have no use for.
	stream >> line;
	if (stream.eof()) {
	  // If we get an EOF on the first frame, then we have an empty movie;
	  // delete the movie from the scene even if we've been asked to
	  // repeat.
	  if (m_firstFrame) {
	    shutDown ();
	  } else if (m_repeat) {
	    stream.clear ();
	    stream.seekg (0);
	    stream >> line;
	    if (stream.eof ()) {
	      // EOF at the beginning?  Give up.
	      needShutdown = true;
	    }
	  } else {
	    needShutdown = true;
	  }
	}
      }
      if (m_playing && !needShutdown) {
	bool foundValidObject = false;
	String error;
	// Number of atoms, and three other numbers.
	stream >> line;
	int nAtoms = line.toInt (0, &error);
	if (error.size()) {
	  message ("Failed to parse the number of atoms from "+
		   m_fileName + ": "+error);
	} else if (nAtoms < 0) {
	  error = "error";
	  message ("Negative number of atoms in "+m_fileName);
	} else if (!m_firstFrame && nAtoms != m_links.size()) {
	  message (String ("Number of atoms changed from ")+m_links.size()+
		   " to "+ nAtoms +" in "+m_fileName);
	  initMovieState ();
	}
	check ();
	// Timestep and time; don't need it.
	stream >> line;
	// Periodicity of space; don't need it.
	stream >> line;
	for (int i = 0; i < nAtoms && 0 == error.size(); i++) {
	  stream >> line;
	  if (stream.eof ()) {
	    // Premature eof isn't an error, since we are likely to be
	    // animating from a movie that's being written as we read it.
	    // Say we found a valid object, since we didn't look at all of the
	    // objects and are therefore likely to miss valid objects that are
	    // there.
	    foundValidObject = true;
	    break;
	  }
	  int atomNo;
	  if (!error.size()) {
	    atomNo = line.toInt (0, &error);
	    if (0 != error.size()) {
	      message ("Failed to parse this line from "+m_fileName+": "
		       + line);
	    }
	  }
	  if (!error.size ()) {
	    if (atomNo != i+1) {
	      error = "error";
	      message (String ("Out-of-sequence atomic number ")+atomNo
		       + " in "+m_fileName + "at this line: " + line);
	    }
	  }
	  int atomicNo = 0;
	  Vec3 where;
	  if (!error.size ()) {
	    atomicNo = line.toInt (5, &error);
	    where [0] = line.toFloat (11, &error);
	    where [1] = line.toFloat (31, &error);
	    where [2] = line.toFloat (51, &error);
	    if (error.size()) {
	      message ("Failed to parse this line from "+m_fileName+": "
		       + line);
	    }
	  }
	  if (!error.size()) {
	    if (atomicNo < 1 || atomicNo >= AtomInfo::maxNumber) {
	      error = "error";
	      message (String ("Out-of-bounds atomic number ")+atomicNo);
	    }
	  }
	  if (m_firstFrame) {
	    m_offsets.push (Vec3 (0,0,0));
	    m_lastPosition.push (Vec3 (0, 0, 0));
	    m_AtomConfiguration->setSymbolNumber (atomicNo);
	    SP<BoringAtom> ba =
	      static_cast <BoringAtom *>
	      (&*m_boringAtomFactory->makeIt (&*m_AtomConfiguration));
	    ba->plausibleState (m_plausibleState);
	    OL<BoringAtom> newLink (sg, ba, &*m_plausibleState);
	    m_links.push (newLink);
	  }
	  if (m_links [i].isValid ()) {
	    foundValidObject = true;
	    Vec3 oldWhere = m_links[i].boundingSphereCenter ();
	    if (oldWhere != m_lastPosition [i]) {
	      m_offsets[i] += oldWhere - m_lastPosition [i];
	    }
	    m_lastPosition [i] = where + m_offsets [i];
	    BoringAtom::setCenter (m_links[i].objectState (),
				   m_lastPosition [i]);
	  }
	}
	if (!foundValidObject) {
	  needShutdown = true;
	}
	m_firstFrame = false;
      }
      if (needShutdown) {
	shutDown ();
	// this may be deallocated at this point, so no code allowed after this
	// point.
      }
    }
  };
  class PlayBrennerMovie
    : public TypedFactory <PlayBrennerMovieConfiguration, Action>
  {
  public:
    PlayBrennerMovie ()
      : TypedFactory <PlayBrennerMovieConfiguration, Action> ("PlayBrennerMovie")
    {}
    SP<PlayBrennerMovieConfiguration> typedDefaultConfiguration () const {
      SP<PlayBrennerMovieConfiguration> result =
	NEW (PlayBrennerMovieConfiguration ());
      result->setFileName ("md/xmol.d");
      return result;
    }
    SP<Action> makeIt (PlayBrennerMovieConfiguration *conf) const {
      SP<Action> result = NEW (Action ());
      String fileName = conf->getFileName ();
      SP<MoviePlayer> mp = NEW (MoviePlayer (fileName, conf->getRepeat ()));
      if (0 != mp->getError().size()) {
	result->setProblem (mp->getError());
      } else {
	SP<TopLevel> top = conf->getTopLevel ();
	SP<SceneGraph> sg = top->getSceneGraph ();
	sg->addObject (mp, 0);
      }
      return result;
    }
  };
  const bool useless =
  (FactoryTable::store ("FileSceneLoader", NEW (PlayBrennerMovie ())),
   true);
}
