// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __FaceManip_h__
#define __FaceManip_h__

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

class LinkManager;
class SceneGraph;
class ObjectLink;
class FaceManip {
public:
  // Do depth-first-search over the links scene graph (as defined in the
  // LinkManager lm), starting at from, for the node to.  maxDepth is how deep
  // to be willing to go.  excludeList is a bunch of nodes that we shouldn't
  // permit to be on the path from "from" to "to", although if either "from" or
  // "to" themselves are on excludeList, that won't prevent us from finding a
  // path.  The returned boolean value is true if we found a path, and if the
  // actualDepth pointer is non-null, it will be filled in with the depth that
  // at which we actually found something.
  //
  // To avoid off-by-ones with maxDepth and actualDepth, here's an example to
  // extrapolate from:
  // We have the points A B C D E, each connected only to the next.
  // A and B are on the excludeList, and it doesn't matter if E is on the
  // excludeList.
  // from is B.
  // to is E.
  // if maxDepth is 3 or more, we'll return true, and fill in actualDepth to be
  // 3.
  static bool searchGraph (const LinkManager *lm, int from, int to,
			   Dynavec <int> &excludeList, int maxDepth,
			   int *actualDepth = 0);
  // Typically, when a new face is added to a figure, some nodes (atoms) may
  // have to be identified with each other to sew the face into the figure
  // properly.  For instance, if you start with a hexagon, and glue a new
  // hexagon onto each edge of the old hexagon, then you'll have four atoms
  // bonded to each corner of the original hexagon, and you want to identify
  // the two new ones with each other so the resulting figure is sensible.
  // 
  // edgeTrim does this.  Give it the scene graph "sg" and give it "start" as
  // the atom that may have four atoms bound to it.  If "start" actually does
  // have four atoms bound to it, and two of them only are on one
  // "reasonable-sized" face (as defined below) that includes "start", then
  // we'll identify those two, and the one that survives the identification
  // will be put in "start" and we'll try again and keep going until the we
  // don't have any more identifications to do.  We return true if we did any
  // work, and we update "start" in place to point to the surviving node of the
  // last identification we did.  (The caller might want to leave that node
  // selected.)
  //
  // Here "reasonable-sized" is determined by calling searchGraph with a
  // maximum depth of 5, starting some node adjacent to the one we are thinking
  // of identifying, after puting start and the node we are considering
  // identifying on the exclude list.  This means that a face of size 7 or less
  // is a face good for sewing together, and a face of 8 or more is considered
  // to be an open place that we shouldn't do anything with.
  //
  // For the example with gluing hexagons to the edges of a hexagon, you'll
  // want to call edgeTrim with each corner of the old hexagon.  If you only
  // glued a hexagon to one edge of the old hexagon, you'd want to call
  // edgeTrim with both ends of that edge, since you want to try sewing the
  // face on from both directions.
  //
  // maxpoly is the maximum number of sides that edgeTrim will recognize as a
  // polygon. 
  static bool edgeTrim (SceneGraph *sg, int start, int maxpoly = 8);
};

#endif
