// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __FaceManip_h__
#include "FaceManip.h"
#endif

#ifndef __LinkManager_h__
#include "LinkManager.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __VecUtil_h__
#include "VecUtil.h"
#endif

#ifndef __SceneGraph_h__
#include "SceneGraph.h"
#endif

#ifndef __Identify_h__
#include "Identify.h"
#endif

#ifndef __ObjectLink_h__
#include "ObjectLink.h"
#endif

bool FaceManip::searchGraph (const LinkManager *lm, int from, int to,
			     Dynavec <int> &excludeList, int maxDepth,
			     int *actualDepth = 0) {
  if (from == to) {
    if (actualDepth) {
      *actualDepth = 0;
    }
    return true;
  } else if (maxDepth <= 0) {
    return false;
  } else {
    int count;
    const int *links;
    lm->getLinks (from, count, links);
    bool result = false;
    excludeList.push (from);
    for (int i = 0; i < count; i++) {
      const int li = links [i];
      if (li == to) {
	// If we found it, that's a win, even if it's on the excludeList.
	if (actualDepth) {
	  *actualDepth = 1;
	}
	result = true;
	break;
      } else if (-1 != li && ! VecUtil::find (li, excludeList)) {
	if (searchGraph (lm, li, to, excludeList, maxDepth - 1, actualDepth)) {
	  if (actualDepth) {
	    (*actualDepth) ++;
	  }
	  result = true;
	  break;
	}
      }
    }
    excludeList.pop ();
    return result;
  }
}

bool FaceManip::edgeTrim (SceneGraph *sg, int start, int maxpoly) {
  bool didWork = false;
  SP<const LinkManager> lm = &*sg->getLinkManager ();
  for (;;) {
    Dynavec <int> links = lm->simpleLinks (start);
    if (links.size() != 4) {
      break;
    } else {
      Dynavec <int> oneFaces;
      for (int j = 0; j < links.size(); j++) {
	const int sj = links [j];
	Dynavec <int> excludeList = Dynavec <int> (start, sj);
	Dynavec <int> jLinks = lm->simpleLinks (sj);
	int faces = 0;
	for (int k = 0; k < jLinks.size(); k++) {
	  if (jLinks [k] != start &&
	      // Find 5 nodes or fewer other than jLinks[k], sj, and start.
	      // This will find octagons or less.  There's a weirdness, I don't
	      // know yet whether it's a problem -- two pentagons that share an
	      // edge have 8 nodes, or an octagon, around their perimeter.
	      FaceManip::searchGraph (lm, jLinks[k], start,
				      excludeList, maxpoly - 2)) {
	    // If it's an octagon or less, add it.
	    faces++;
	  }
	}
	// Now faces is the number of faces that include the start-sj edge.
	if (1 == faces) {
	  oneFaces.push (sj);
	}
      }
      if (2 == oneFaces.size()) {
	// If there are exactly two nodes adjacent to start that have one face
	// on their edge with start, identify those two nodes, and continue the
	// loop with start as the node we identified so we can sew the entire
	// perimeter of the face into the scene if it needs it.
	start = oneFaces [1];
	Identify::identify (sg, oneFaces [0], oneFaces [1]);
	didWork = true;
      } else {
	break;
      }
    }
  }
  return didWork;
}
