// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __EdgeHexConfiguration_h__
#include "EdgeHexConfiguration.h"
#endif

#ifndef __Action_h__
#include "Action.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __Dynavec_h__
#include "Dynavec.h"
#endif

#ifndef __TopLevel_h__
#include "TopLevel.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __LinkManager_h__
#include "LinkManager.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __SceneGraph_h__
#include "SceneGraph.h"
#endif

#ifndef __PhysicsObject_h__
#include "PhysicsObject.h"
#endif

#ifndef __TypedFactory_h__
#include "TypedFactory.h"
#endif

#ifndef __SelectionManager_h__
#include "SelectionManager.h"
#endif

#ifndef __BoringAtom_h__
#include "BoringAtom.h"
#endif

#ifndef __RecursiveSlotValue_h__
#include "RecursiveSlotValue.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __FaceManip_h__
#include "FaceManip.h"
#endif

#ifndef __ObjectLink_h__
#include "ObjectLink.h"
#endif

#ifndef __Canvas_h__
#include "Canvas.h"
#endif

#ifndef __KeyboardMouseEvent_h__
#include "KeyboardMouseEvent.h"
#endif

namespace {
  class EdgeHex
    : public TypedFactory <EdgeHexConfiguration, Action>
  {
    // Make a simple polygon.
    void makeSimplePoly (TopLevel *top, int numAtoms,
			 BoringAtom *atom,
			 Dynavec <ObjectLink> &endSelection) const
    {
      endSelection.extendTo (0);
      SP<Canvas> canvas = top->getCanvas ();
      SP<SceneGraph> sg = top->getSceneGraph ();
      SP<LinkManager> lm = sg->getLinkManager ();
      Vec3 screenWhere =
	canvas->transformWorldToRHScreen (sg->centerOfSomething ());
      SP<const KeyboardMouseEvent> e =
	dynamic_cast <const KeyboardMouseEvent *> (&*top->getEvent ());
      assert (e);
      screenWhere [0] = e->getX ();
      screenWhere [1] = e->getY ();
      // DesignAtoms will move around to a more plausible place.
      const Float radius = canvas->getScale ();
      Dynavec <Float> plausibleState;
      atom->plausibleState (plausibleState);
      BoringAtom::setCenter (&*plausibleState,
			     canvas->transformRHScreenToWorld
			     (screenWhere + Vec3 (radius, 0, 0)));
      const int firstAtom = sg->addObject (atom, &*plausibleState);
      endSelection.push (ObjectLink (sg, firstAtom));
      int previousAtom = firstAtom;
      const Float angle = 2.0 * FloatUtil::PI / numAtoms;
      for (int i = 1; i < numAtoms; i++) {
	BoringAtom::setCenter
	  (&*plausibleState,
	   canvas->transformRHScreenToWorld
	   (screenWhere + Vec3 (radius * cos (i * angle),
				radius * sin (i * angle),
				0)));
	const int nextAtom = sg->addObject (atom, &*plausibleState);
	if (-1 != nextAtom) {
	  endSelection.push (ObjectLink (sg, nextAtom));
	  if (-1 != previousAtom) {
	    lm->addNewLink (previousAtom, nextAtom);
	    lm->addNewLink (nextAtom, previousAtom);
	  }
	}
	previousAtom = nextAtom;
      }
      lm->addNewLink (previousAtom, firstAtom);
      lm->addNewLink (firstAtom, previousAtom);
    }
  public:
    EdgeHex ()
      : TypedFactory <EdgeHexConfiguration, Action> ("EdgeHex")
    {}
    SP<EdgeHexConfiguration> typedDefaultConfiguration () const {
      return NEW (EdgeHexConfiguration ());
    }
    SP<Action> makeIt (EdgeHexConfiguration *conf) const {
      // Number of atoms to add around the perimeter of a face.
      const int numAtoms = conf->getNewFaceSize () - 2;
      // The largest polygon to recognize as a face.
      const int maxPoly = conf->getOldFaceSize ();
      const bool selectPoints = conf->getSelectPoints ();
      SP<TopLevel> top = conf->getTopLevel ();
      SP<SceneGraph> sg = top->getSceneGraph ();
      SP<LinkManager> lm = sg->getLinkManager ();
      Dynavec <ObjectLink> selection;
      // All of the nodes that existed when we started, so we can deselect them
      // at the end.
      Dynavec <ObjectLink> initialNodes;
      for (int i = 0; i < sg->maxIndex(); i++) {
	initialNodes.push (ObjectLink (sg, i));
      }
      SelectionManager::getSelectionLinks (selection);
      // We accumulate the things to select when we're done in endSelection.
      Dynavec <ObjectLink> endSelection;
      SP<BoringAtom> ba =
	dynamic_cast <BoringAtom *> (&*conf->getAtomFactory()->makeIt());
      if (0 == selection.size ()) {
	makeSimplePoly (top, conf->getNewFaceSize (), ba, endSelection);
      } else {
	// If selectPoints is set, then the edges we want to consider are edges
	// between points selected at the beginning of this function and their
	// neighbors that existed at the beginning of this function.
	// Wrong way to do it #1: If instead we consider all edges between nodes
	// that are selected at the beginning and their neighbors at the time we
	// are thinking of adding a node, then we can add new faces to border new
	// faces we added, which is too much.
	// Wrong way to do it #2: If instead we add all neighbors of selected
	// points to the selection, and then consider all edges between pairs of
	// selected points, then we can add faces to edges that join neighbors of
	// two different initially-selected points, even when those two neighbors
	// are not themselves initially-selected, which is too much again.
	// If selectPoints is set, then goodNeighbors will end up with all
	// neighbors of selected points that exist at the beginning of the game.
	Dynavec <ObjectLink> goodNeighbors;
	if (selectPoints) {
	  for (int i = 0; i < selection.size (); i++) {
	    Dynavec <int> intLinks =
	      lm->simpleLinks (selection[i].getIndex ());
	    for (int j = 0; j < intLinks.size (); j++) {
	      goodNeighbors.push (ObjectLink (sg, intLinks [j]));
	    }
	  }
	} else {
	  // If we're taking any edge between two selected points, then all
	  // selected points are candidate neighbors.
	  goodNeighbors = selection;
	}
	for (int i = 0; i < selection.size(); i++) {
	  ObjectLink si = selection [i];
	  Dynavec <ObjectLink> siLinks;
	  {
	    Dynavec <int> intsiLinks = lm->simpleLinks (si.getIndex());
	    for (int z = 0; z < intsiLinks.size(); z++) {
	      siLinks.push (ObjectLink (sg, intsiLinks [z]));
	    }
	  }
	  for (int j = 0; j < siLinks.size(); j++) {
	    ObjectLink sj = siLinks [j];
	    // We used to require sj > si to avoid considering each link twice,
	    // but that's wrong when things can get renumbered on the fly, so
	    // instead just consider each link twice.
	    bool isSjGood = false;
	    for (int k = 0; k < goodNeighbors.size(); k++) {
	      if (goodNeighbors[k] == sj) {
		isSjGood = true;
		break;
	      }
	    }
	    if (isSjGood) {
	      // si to sj is a link we want to add hexes around.
	      // Count the number of faces that include the si-sj link.  A face
	      // is a closed polygon of length 8 or less.
	      int faces = 0;
	      {
		// Compute excludeList inside the loop because nodes may get
		// renumbered by edgeTrim.
		Dynavec <int> excludeList;
		excludeList.push (si.getIndex ());
		for (int k = 0; k < siLinks.size(); k++) {
		  excludeList.push (siLinks [k].getIndex ());
		}
		// excludeList now has si and all of its direct neighbors and all
		// of the new nodes.
		for (int k = 0; k < siLinks.size (); k++) {
		  if (siLinks [k] != sj &&
		      FaceManip::searchGraph (lm, siLinks[k].getIndex(),
					      sj.getIndex (),
					      excludeList, maxPoly - 2)) {
		    // We can find sj by searching to depth 6 or less from
		    // links[k].  That means we have a face of size 6 + 1 for
		    // links[k] + 1 for si = 8 edges or less bordering the si-sj
		    // edge.
		    faces++;
		  }
		}
	      }
	      if (faces < 2) {
		// Add a polygon face.
		// First, guess a direction to insert the face that will push it
		// generally away from the stuff near the edge we're working
		// with.
		int everybodyCount = 2;
		// everybodyCenter will eventually be the average position of
		// everybody linked to si or sj.
		Vec3 everybodyCenter =
		  si.boundingSphereCenter () + sj.boundingSphereCenter ();
		Vec3 edgeCenter = everybodyCenter / 2;
		for (int k = 0; k < siLinks.size(); k++) {
		  if (siLinks [k] != sj && &*(siLinks [k])) {
		    everybodyCenter += siLinks [k].boundingSphereCenter ();
		    everybodyCount++;
		  }
		}
		{
		  Dynavec <int> sjLinks = lm->simpleLinks (sj.getIndex ());
		  for (int k = 0; k < sjLinks.size (); k++) {
		    if (sjLinks [k] != si.getIndex ()
			&& sg->object (sjLinks [k])) {
		      everybodyCenter += sg->boundingSphereCenter (sjLinks [k]);
		      everybodyCount++;
		    }
		  }
		}
		everybodyCenter = everybodyCenter / everybodyCount;
		Vec3 away = edgeCenter - everybodyCenter;
		const Vec3 edgeVec =
		  sj.boundingSphereCenter () - si.boundingSphereCenter ();
		if (away.length () < 0.01) {
		  // Too short, just pick any direction 
		  away = edgeVec.anyVectorNormalTo ();
		}
		// Fix up away so that it is normal to edgeVec, and has unit
		// length.
		if (edgeVec.length() < 0.01) {
		  // si and sj are right on top of each other, so there's no
		  // hope to get it right.
		  away = away.makeUnit ();
		} else {
		  const Vec3 edgeVecUnit = edgeVec.makeUnit ();
		  const Vec3 parallelComponent =
		    (edgeVecUnit * away) * edgeVecUnit;
		  const Vec3 perpendicularComponent = away - parallelComponent;
		  away = perpendicularComponent.makeUnit ();
		}
		const Vec3 awayEnd1 = edgeCenter + away + (edgeVec.makeUnit());
		const Vec3 awayEnd2 = edgeCenter + away - (edgeVec.makeUnit());
		// Now awayEnd1 and awayEnd2 are the ends of a line that we'll
		// put our new atoms on.  We already have two atoms for our new
		// face, si and sj.
		int previousAtom = sj.getIndex ();
		for (int i = 0; i < numAtoms; i++) {
		  const Vec3 newPos =
		    awayEnd1 +
		    (awayEnd2 - awayEnd1) * ((Float) i) / (numAtoms - 1);
		  Dynavec <Float> plausibleState;
		  ba->plausibleState (plausibleState);
		  BoringAtom::setCenter (&*plausibleState, newPos);
		  const int newAtom = sg->addObject (ba, &*plausibleState);
		  if (-1 != newAtom) {
		    endSelection.push (ObjectLink (sg, newAtom));
		    if (-1 != previousAtom) {
		      lm->addNewLink (previousAtom, newAtom);
		      lm->addNewLink (newAtom, previousAtom);
		    }
		  }
		  previousAtom = newAtom;
		}
		lm->addNewLink (si.getIndex (), previousAtom);
		lm->addNewLink (previousAtom, si.getIndex());
		// Now we have the new face.  Do an edgeTrim.
		FaceManip::edgeTrim (sg, si.getIndex (), maxPoly);
		FaceManip::edgeTrim (sg, sj.getIndex (), maxPoly);
	      }
	    }
	  }
	}
	// Remove all of the originally existing nodes from the new selection.
	// We can have some of them in there because they can get identified
	// with new nodes during edgeTrim.
	{
	  Dynavec <bool> allNodes;
	  allNodes.extendTo (sg->maxIndex (), false);
	  for (int i = 0; i < initialNodes.size (); i++) {
	    allNodes [initialNodes[i].getIndex ()] = true;
	  }
	  int i = 0;
	  for (;;) {
	    if (i >= endSelection.size ()) break;
	    if (allNodes [endSelection [i].getIndex ()]) {
	      if (endSelection.size() - 1 > i) {
		endSelection [i] = endSelection [endSelection.size()-1];
	      }
	      endSelection.extendTo (endSelection.size() - 1);
	    } else {
	      i++;
	    }
	  }
	}
      }
      SelectionManager::setSelectionLinks (sg, endSelection);
      return NEW (Action ());
    }
  };
  const bool useless =
  // This is a KeyboardMouseAction instead of a KeyboardAction only because of
  // the behavior when nothing is selected.  It creates a polygon near the
  // mouse.
  (FactoryTable::store ("KeyboardMouseAction", NEW (EdgeHex ())),
   false);
};
