// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __BrennerAtom_h__
#include "BrennerAtom.h"
#endif

#ifndef __TypedFactory_h__
#include "TypedFactory.h"
#endif

#ifndef __AtomConfiguration_h__
#include "AtomConfiguration.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __AtomInfo_h__
#include "AtomInfo.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __PhysicsObjectInfo_h__
#include "PhysicsObjectInfo.h"
#endif

#ifndef __FloatUtil_h__
#include "FloatUtil.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __AtomCache_h__
#include "AtomCache.h"
#endif

namespace {
  // If a design atom is not linked to another design atom, it will try to push
  // that atom away to designAtomRadius angstroms.  If this is too big, then
  // neighbor computation is slow.  If this is too small, then you don't get
  // the functionality you want because they'll clump too much.
  // 
  // DesignAtom's don't have momentum.  The force is interpreted directly as
  // the derivative of position, so things don't bounce around too much.
  // FIXME If I set designAtomRadius to 6, it doesn't work at all.
  // Things are too far apart, probably due to the maximum attractive force 
  const Float designAtomRadius = 3;
  // Equilibrium bond lengths for design atoms.  We want these to be close
  // enough so that if we do a minimization pass with Brenner's code, it will
  // adjust things so they are right.  In general, too small numbers are better
  // than too large numbers, since small distances will cause atoms to push
  // apart to the correct distance, but large distances may cause things to
  // fail to interact at all and then we get the wrong minimum.
  const Float designHH = 0.75;
  const Float designHC = 1.12;
  const Float designCC = 1.42;
  // headRoom is the distance that we can permit our force field to extend
  // beyond the equilibrium distance.  We want a safe distance for all
  // equilibrium distances, which means we want to subtract out the largest
  // equilibrium distance, which happens to be designCC.
  const Float headRoom = designAtomRadius - designCC;
  const int designDistTabSize = 7;
  const Float designDistTab [designDistTabSize] [designDistTabSize] =
  {{0, 0,        0, 0, 0, 0, 0},
   {0, designHH, 0, 0, 0, 0, designHC},
   {0, 0,        0, 0, 0, 0, 0},
   {0, 0,        0, 0, 0, 0, 0},
   {0, 0,        0, 0, 0, 0, 0},
   {0, 0,        0, 0, 0, 0, 0},
   {0, designHC, 0, 0, 0, 0, designCC}};
  bool didInit = false;
  // Next table will have the square of the distance between 
  Float maxDist  [designDistTabSize] [designDistTabSize];
  class DesignAtom
    : public BoringAtom
  {
  public:
    DesignAtom (AtomConfiguration *b, const Factory *f)
      : BoringAtom (b, f)
    {
      if (!didInit) {
	didInit = true;
	for (int i = 0; i < designDistTabSize; i++) {
	  for (int j = 0; j < designDistTabSize; j++) {
	    if (0 == designDistTab [i] [j]) {
	      maxDist [i] [j] = 0;
	    } else {
	      const Float f = designDistTab [i] [j] + headRoom;
	      maxDist [i] [j] = f * f;
	    }
	  }
	}
      }
    }
    Float physicsBoundingSphereRadius () const {
      return designAtomRadius;
    };
    static const char *const staticClassId;
    // We've adjusted dist so that the equilibrium distance is 0.  Return a
    // positive force that has a continuous fourth derivative (to make
    // RungaKutta happy) and peters down to 0 at dist = headRoom.
    // According to gcov, about 90% of the time this returns 0.
    static Float compForce (const Float dist) {
      const Float headDist = headRoom - dist;
      assert (headDist >= 0);
      const Float d2 = headDist * headDist;
      const Float d4 = d2 * d2;
      return d4 * headDist;
    }
    void doPhysics (const Float *state,
		    Float *deriv,
		    int myIndex,
		    const Dynavec<PhysicsObjectInfo> &neighbors,
		    const Dynavec<PhysicsObjectInfo> &links) const;
    bool isRelevantNeighbor (const PhysicsObject *o, int myIndex, int neighborIndex) const;
    const char *classId () const {
      return staticClassId;
    }
    bool isInstanceOf (const char *classId) const {
      return classId == staticClassId || BoringAtom::isInstanceOf (classId);
    }
  };
  bool DesignAtom::isRelevantNeighbor (const PhysicsObject *o, int myIndex, int neighborIndex)
    const
  {
    const bool isDA = o->classId() == DesignAtom::staticClassId;
    if ((isDA & (neighborIndex > myIndex)) ||
	((!isDA) && o->isInstanceOf (BoringAtom::staticClassId))) {
      const BoringAtom *da = static_cast <const BoringAtom *> (o);
      const int toSym = da->getSymbolNumber ();
      assert (toSym > 0);
      if (toSym < designDistTabSize) {
	return true;
      } else {
	return false;
      }
    } else {
      return false;
    }
  }
  void DesignAtom::doPhysics (const Float *state,
			      Float *deriv,
			      int myIndex,
			      const Dynavec<PhysicsObjectInfo> &neighbors,
			      const Dynavec<PhysicsObjectInfo> &links) const {
    const int fromSym = getSymbolNumber ();
    const Vec3 myCenter = center (state + myIndex);
    assert (fromSym > 0);
    if (fromSym >= designDistTabSize) return;
    // Essentially all the time happens in the following loop, because we have
    // hundreds of nearby atoms but few links.
    for (int i = 0; i < neighbors.size(); i++) {
      PhysicsObjectInfo poi = neighbors [i];
      assert (poi.m_object);
#ifndef NDEBUG
      const bool isDA = poi.m_object->classId() == DesignAtom::staticClassId;
#endif
      assert ((isDA & (poi.m_index > myIndex)) ||
	      ((!isDA) &&
	       poi.m_object->isInstanceOf (BoringAtom::staticClassId)));
      const BoringAtom *da =
	static_cast <const BoringAtom *> (poi.m_object);
      const int toSym = da->getSymbolNumber ();
      assert (toSym > 0);
      // Vector from me to the other guy.
      const Vec3 vec = BoringAtom::center (state + poi.m_index) -
	myCenter;
      const Float distSquared = vec.lengthSquared();
      const Float maxD = maxDist [fromSym] [toSym];
      if ((distSquared < maxD) && (distSquared >= 0.01 * 0.01)) {
	// The purpose of maxD is to avoid taking the following square root
	// most of the time.
	const Float dist = sqrt (distSquared);
	const Float equilibriumDist = designDistTab [fromSym] [toSym];
	const Float force = compForce (dist - equilibriumDist);
	assert (force > 0);
	const Vec3 forceVec = vec * (force / dist);
	setCenter (deriv+myIndex, center (deriv+myIndex)-forceVec);
	if (poi.m_object->classId() == DesignAtom::staticClassId) {
	  setCenter (deriv+poi.m_index,
		     center (deriv+poi.m_index)+forceVec);
	}
      }
    }
    for (int i = 0; i < links.size(); i++) {
      PhysicsObjectInfo poi = links [i];
      // poi.m_object can be null.  Here's the scenario: create an atom, make
      // another atom, link them, delete one of them.  The link index that the
      // other one has in its list has to become -1 because we support links
      // with meaning based on position, and the corresponding m_object ought
      // to be 0 because there's no other good value.
      if (0 != poi.m_object) {
	const bool isDA = poi.m_object->classId() == DesignAtom::staticClassId;
	if ((isDA & (poi.m_index > myIndex)) |
	    ((!isDA) & poi.m_object->isInstanceOf (BoringAtom::staticClassId))) {
	  const BoringAtom *da = static_cast <const BoringAtom *> (poi.m_object);
	  const int toSym = da->getSymbolNumber ();
	  assert (toSym > 0);
	  if (toSym < designDistTabSize) {
	    const Float equilibriumDist = designDistTab [fromSym] [toSym];
	    if (equilibriumDist > 0) {
	      // Vector from me to the other guy.
	      const Vec3 vec =
		DesignAtom::center (state + poi.m_index) - myCenter;
	      const Float dist = vec.length();
	      if (dist > 0.01) {
		// Use min here because the distance can be quite large, and we don't
		// want to get astronomical velocities and broken numerical
		// integration in that case.
		const Float force =
		  FloatUtil::min ((Float)30.0, compForce (equilibriumDist - dist));
		assert (force >= 0);
		const Vec3 forceVec = vec * (force / dist);
		setCenter (deriv+myIndex, center (deriv+myIndex)+forceVec);
		if (isDA) {
		  setCenter (deriv+poi.m_index,
			     center (deriv+poi.m_index)-forceVec);
		}
	      }
	    }
	  }
	}
      }
    }
  }
  const char *const DesignAtom::staticClassId = "DesignAtom";
  class DesignAtomFactory
    : TypedFactory <AtomConfiguration, DesignAtom>
  {
    static SP<AtomCache <DesignAtom> > s_cache;
    static void deallocate () {
      if (s_cache) {
	s_cache = 0;
      }
    }
    static const bool s_useless;
  public:
    DesignAtomFactory ()
      : TypedFactory <AtomConfiguration, DesignAtom> ("DesignAtom")
    {}
    SP<AtomConfiguration> typedDefaultConfiguration () const {
      return NEW (AtomConfiguration ());
    }
    SP<DesignAtom> makeIt (AtomConfiguration *ai) const {
      SP<DesignAtom> &da = s_cache->cacheref (ai);
      if (!da) {
	da = NEW (DesignAtom (ai, this));
      }
      return da;
    }
  };
  SP<AtomCache<DesignAtom> > DesignAtomFactory::s_cache =
  NEW (AtomCache <DesignAtom> ());
  const bool DesignAtomFactory::s_useless =
  (FactoryTable::store("Atom", NEW (DesignAtomFactory ())),
   MemoryUtil::registerDeallocator (deallocate),
   true);
}
