// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __TypedFactory_h__
#include "TypedFactory.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __ChangeAtomTypeConfiguration_h__
#include "ChangeAtomTypeConfiguration.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __SP_h__
#include "SP.h"
#endif

#ifndef __SceneGraph_h__
#include "SceneGraph.h"
#endif

#ifndef __TopLevel_h__
#include "TopLevel.h"
#endif

#ifndef __SelectionManager_h__
#include "SelectionManager.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __Action_h__
#include "Action.h"
#endif

#ifndef __RecursiveSlotValue_h__
#include "RecursiveSlotValue.h"
#endif

#ifndef __AtomConfiguration_h__
#include "AtomConfiguration.h"
#endif

#ifndef __LinkManager_h__
#include "LinkManager.h"
#endif

#ifndef __BoringAtom_h__
#include "BoringAtom.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

namespace {
  class ChangeAtomType
    : public TypedFactory <ChangeAtomTypeConfiguration, Action>
  {
  public:
    ChangeAtomType ()
      : TypedFactory <ChangeAtomTypeConfiguration, Action> ("ChangeAtomType")
    {}
    SP<ChangeAtomTypeConfiguration> typedDefaultConfiguration () const {
      return NEW (ChangeAtomTypeConfiguration ());
    }
    SP<Action> makeIt (ChangeAtomTypeConfiguration *c) const {
      Dynavec <int> selection;
      SelectionManager::getSelection (selection);
      SP<Action> result = NEW (Action ());
      if (!selection.size ()) {
	result->setProblem ("Select something first.");
      }
      Dynavec <int> oldToNew;
      SP<SceneGraph> sg;
      SP<RecursiveSlotValue> rsv;
      SP<AtomConfiguration> ai;
      if (!result->isProblem ()) {
	sg = c->getTopLevel ()->getSceneGraph ();
	oldToNew.extendTo (sg->maxIndex (), -1);
	// Make a copy here so we don't edit the original configuration.  The
	// original configuration typically shares with other configurations,
	// and we will be setting the atom number below.  We don't want to
	// overwrite the user's preferred atom number with whatever number we
	// come up with here.
	rsv = dynamic_cast <RecursiveSlotValue *> (&*c->getAtomSlot ()->copy());
	assert (rsv);
	ai = dynamic_cast <AtomConfiguration *> (&*rsv->getConfiguration ());
	if (!ai) {
	  result->setProblem ("The configuration for the factory given to "
			      "ChangeAtomType is not an AtomConfiguration");
	}
      }
      SP<LinkManager> lm;
      if (!result->isProblem ()) {
	lm = sg->getLinkManager ();
	Dynavec <Float> newState;
	Dynavec <int> newLinks;
	for (int i = 0; i < selection.size(); i++) {
	  const int s = selection [i];
	  // Would be slightly faster if I had used the staticClassId and then
	  // a static_cast, I think.  Doesn't matter, this doesn't have to be
	  // blindingly fast anyway.
	  SP<const BoringAtom> const ba =
	    dynamic_cast <const BoringAtom *> (&*sg->object (s));
	  if (ba) {
	    const Float *const oldState = sg->objectState (s);
	    ai->setSymbolNumber (ba->getSymbolNumber ());
	    SP<const PhysicsObject> po =
	      dynamic_cast <const PhysicsObject *> (&*rsv->makeIt ());
	    assert (po);
	    assert (dynamic_cast <const BoringAtom *> (&*po));
	    po->plausibleState (newState);
	    BoringAtom::setCenter (&*newState, BoringAtom::center (oldState));
	    BoringAtom::setVelocity (&*newState,
				     BoringAtom::velocity (oldState));
	    oldToNew [s] = sg->addObject (po, newState);
	    int count;
	    const int *links;
	    lm->getLinks (s, count, links);
	    if (count > 0) {
	      newLinks.extendTo (0);
	      for (int j = 0; j < count; j++) {
		newLinks.push (links [j]);
	      }
	      lm->setLinks (oldToNew [s], newLinks);
	    }
	  }
	}
	// We've made new objects, so sg->maxIndex () is probably bigger.
	// Extend oldToNew with -1's.
	oldToNew.extendTo (sg->maxIndex(), -1);
	// Redirect the links to the objects we're about to delete.
	for (int i = 0; i < sg->maxIndex(); i++) {
	  int count;
	  const int *links;
	  lm->getLinks (i, count, links);
	  bool needRedirect = false;
	  for (int j = 0; j < count; j++) {
	    const int l = links [j];
	    if (l != -1 && oldToNew [l] != -1) {
	      needRedirect = true;
	      break;
	    }
	  }
	  if (needRedirect) {
	    newLinks.extendTo (0);
	    for (int j = 0; j < count; j++) {
	      const int l = links [j];
	      if (l != -1 && oldToNew [l] != -1) {
		newLinks.push (oldToNew [l]);
	      } else {
		newLinks.push (l);
	      }
	    }
	    lm->setLinks (i, newLinks);
	  }
	}
	// Delete the selected objects.
	Dynavec <int> newSelection;
	for (int i = 0; i < selection.size(); i++) {
	  const int s = selection [i];
	  SP<const BoringAtom> ba =
	    dynamic_cast <const BoringAtom *> (&*sg->object (s));
	  if (ba) {
	    assert (-1 != oldToNew [s]);
	    newSelection.push (oldToNew [s]);
	    sg->deleteObject (s);
	  } else {
	    newSelection.push (s);
	  }
	}
	// The renumbering of the links should have caused the replacement
	// objects to be selected.
      }
      return result;
    }
  };
  const bool useless =
  (FactoryTable::store ("Action", NEW (ChangeAtomType ())),
   true);
}
