// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __BrennerAtomConfiguration_h__
#include "BrennerAtomConfiguration.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __IntSlotValue_h__
#include "IntSlotValue.h"
#endif

namespace {
  String motionFlagName () {
    return "Motion Flag (1 = thermostat; 2 = fixed; other = moving)";
  }
}

BrennerAtomConfiguration::BrennerAtomConfiguration () {
  initializeSlot (motionFlagName (), NEW (IntSlotValue (1)));
}

void BrennerAtomConfiguration::setMotionFlag (MotionFlag flag) {
  IntSlotValue::setSlot (this, motionFlagName(), (int) flag);
}

BrennerAtomConfiguration::MotionFlag BrennerAtomConfiguration::getMotionFlag ()
  const
{
  MotionFlag result =
    (MotionFlag) IntSlotValue::getSlot (this, motionFlagName ());
  if (THERMOSTAT != result && RIGID != result) {
    result = DYNAMIC;
  }
  return result;
}

int BrennerAtomConfiguration::getDifferentAtoms () const {
  return AtomConfiguration::getDifferentAtoms () * (((int) LAST) - 1);
}

int BrennerAtomConfiguration::myArrayPos () const {
  return AtomConfiguration::myArrayPos () +
    getMotionFlag () * AtomConfiguration::getDifferentAtoms ();
}


SP<Configuration> BrennerAtomConfiguration::copy () const {
  // Default copy constructor is fine.
  return NEW (BrennerAtomConfiguration (*this));
}

bool BrennerAtomConfiguration::hasTheSameTypeAs (const Configuration *c)
  const {
  return 0 != dynamic_cast <const BrennerAtomConfiguration *> (c);
}

String BrennerAtomConfiguration::unParse () const {
  String result = AtomConfiguration::unParse ();
  result += ",";
  switch (getMotionFlag ()) {
  case DYNAMIC:
    result += "DYNAMIC";
    break;
  case THERMOSTAT:
    result += "THERMOSTAT";
    break;
  case RIGID:
    result += "RIGID";
    break;
  default:
    result += String ((int) getMotionFlag ()) + " (corrupt!)";
    break;
  }
  return result;
}
