// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __BoringAtom_h__
#include "BoringAtom.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __AtomConfiguration_h__
#include "AtomConfiguration.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __TypedFactory_h__
#include "TypedFactory.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __FactoryTable_h__
#include "FactoryTable.h"
#endif

#ifndef __Factory_h__
#include "Factory.h"
#endif

#ifndef __RecursiveSlotValue_h__
#include "RecursiveSlotValue.h"
#endif

#ifndef __AtomInfo_h__
#include "AtomInfo.h"
#endif

#ifndef __Vec3_h__
#include "Vec3.h"
#endif

#ifndef __Color_h__
#include "Color.h"
#endif

#ifndef __BoundingBox_h__
#include "BoundingBox.h"
#endif

#ifndef __Quaternion_h__
#include "Quaternion.h"
#endif

#ifndef __AtomCache_h__
#include "AtomCache.h"
#endif

struct BoringAtom::BoringAtomData {
  int m_symbolNumber;
  // How to make another atom like this one.
  // This could almost be a SP<const RecursiveSlotValue>, except we have to set
  // the configuration right after making it.
  SP<RecursiveSlotValue> m_rsv;
};

namespace {
  String atomProtocol () {
    return "Atom";
  }
}

BoringAtom::BoringAtom (AtomConfiguration *b, const Factory *f)
  : DDPO (b),
    m_data (NEW (BoringAtomData ())) {
  m_data->m_symbolNumber = b->getSymbolNumber ();
  m_data->m_rsv = NEW (RecursiveSlotValue (atomProtocol (), f));
  m_data->m_rsv->setConfiguration (b->copy ());
}

SP<const RecursiveSlotValue> BoringAtom::getFactory () const {
  return &*m_data->m_rsv;
}

BoringAtom::~BoringAtom () {
  assert (m_data);
  delete m_data;
  m_data = 0;
}

int BoringAtom::getSymbolNumber () const {
  return m_data->m_symbolNumber;
}

int BoringAtom::stateSize () const {
  return 6;
}

void BoringAtom::plausibleState (Dynavec <Float> &result) const {
  result.extendTo (0);
  result.extendTo (6, 0);
  // Might do something here to set a temperature someday.
}

void BoringAtom::setVelocity (Float *state, const Vec3 &v) {
  v.copyToPointer (state + 3);
}

const char *const BoringAtom::staticClassId = "BoringAtom";

const char *BoringAtom::classId () const {
  return staticClassId;
}

bool BoringAtom::isInstanceOf (const char *classId) const {
  return classId == staticClassId;
}

class BoringAtom::BoringAtomFactory
  : public TypedFactory <AtomConfiguration, BoringAtom>
{
  static SP<AtomCache <BoringAtom> > s_cache;
  static void deallocate () {
    if (s_cache) {
      s_cache = 0;
    }
  }
  static const bool s_useless;
public:
  BoringAtomFactory ()
    : TypedFactory <AtomConfiguration, BoringAtom> ("BoringAtom")
  {}
  SP<AtomConfiguration> typedDefaultConfiguration () const {
    return NEW (AtomConfiguration ());
  }
  SP<BoringAtom> makeIt (AtomConfiguration *ai) const {
    SP<BoringAtom> &ba = s_cache->cacheref (ai);
    if (!ba) {
      ba = NEW (BoringAtom (ai, this));
    }
    return ba;
  }
};

SP<AtomCache<BoringAtom> > BoringAtom::BoringAtomFactory::s_cache =
NEW (AtomCache <BoringAtom> ());

const bool BoringAtom::BoringAtomFactory::s_useless =
  (FactoryTable::store (atomProtocol (), NEW (BoringAtomFactory ())),
   MemoryUtil::registerDeallocator (deallocate),
   true);

Vec3 BoringAtom::velocity (const Float *state) {
  return Vec3 (state+3);
}

Vec3 BoringAtom::center (const Float *state) {
  return Vec3 (state);
}

void BoringAtom::setCenter (Float *state, const Vec3 &v) {
  v.copyToPointer (state);
}

bool BoringAtom::physicsIntersect (const Float *state,
				   const BoundingBox &box)
  const
{
  (void) state; (void) box;
  return false;
}

Vec3 BoringAtom::boundingSphereCenter (const Float *state) const {
  return Vec3 (state);
}

Float BoringAtom::physicsBoundingSphereRadius () const {
  return 0;
}

bool BoringAtom::isPhysicsBounded () const {
  return true;
}

void BoringAtom::translate (Float *state, const Vec3 &v) const {
  state [0] += v[0];
  state [1] += v[1];
  state [2] += v[2];
}

void BoringAtom::rotate (Float *state, const Quaternion &q) const {
  (void) state; (void) q;
}

bool BoringAtom::exempt () const {
  return true;
}

