// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __AtomInfo_h__
#include "AtomInfo.h"
#endif

#ifndef __Float_h__
#include "Float.h"
#endif

#ifndef __Color_h__
#include "Color.h"
#endif

#ifndef __myassert_h__
#include "myassert.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

namespace {

  // CPK color scheme from
  // http://www.umass.edu/microbio/chime/explorer/pe_tut.htm on 26 Feb 2000. 

  struct AtomColor {
    int rgb; // Format is 0xrrggbb
    const char *elements; // Two characters per element.
  };
  const AtomColor colors [] = {
    {0xc7c7c7, "C "},
    {0xffffff, "H "},
    {0xef0400, "O "},
    {0x8e8eff, "N "},
    {0xffc728, "S "},
    {0xffa600, "P FEBA"},
    {0x0000ff, "NA"},
    {0x188e18, "MG"},
    {0x797d8e, "ALCATICRMNAG"},
    {0xa62c20, "NICUZNBR"},
    {0x00ff00, "B CL"},
    {0xd7a618, "F SIAU"},
    {0x9e1cef, "I "},
    {0xae2418, "LI"},
    {0xffbec7, "HE"}
  };
  Color hexToColor (int color) {
    return Color (((color & 0xff0000) >> 16) / 255.0,
		  ((color & 0xff00) >> 8) / 255.0,
		  (color & 0xff) / 255.0,
		  1.0);
  }
  const Color allOtherColor = hexToColor (0xff148e);
  // Element list from http://pearl1.lanl.gov/periodic/list1.html on 26 Feb
  // 2000.
  struct TmpElement {
    const char *symbol;
    const char *name;
    int number;
  };
  const TmpElement tmpElements [] = {
    {"AC", "Actinium",89},
    {"MD", "Mendelevium", 101},
    {"AL", "Aluminum", 13},
    {"HG", "Mercury", 80},
    {"AM", "Americium", 95},
    {"MO", "Molybdenum", 42},
    {"SB", "Antimony", 51},
    {"ND", "Neodymium", 60},
    {"AR", "Argon", 18},
    {"NE", "Neon", 10},
    {"AS", "Arsenic", 33},
    {"NP", "Neptunium", 93},
    {"AT", "Astatine", 85},
    {"NI", "Nickel", 28},
    {"BA", "Barium", 56},
    {"NB", "Niobium", 41},
    {"BK", "Berkelium", 97},
    {"N", "Nitrogen", 7},
    {"BE", "Beryllium", 4},
    {"NO", "Nobelium", 102},
    {"BI", "Bismuth", 83},
    {"OS", "Osmium", 76},
    {"BH", "Bohrium", 107},
    {"O", "Oxygen", 8},
    {"B", "Boron", 5},
    {"PD", "Palladium", 46},
    {"BR", "Bromine", 35},
    {"P", "Phosphorus", 15},
    {"CD", "Cadmium", 48},
    {"PT", "Platinum", 78},
    {"CA", "Calcium", 20},
    {"PU", "Plutonium", 94},
    {"CF", "Californium", 98},
    {"PO", "Polonium", 84},
    {"C", "Carbon", 6},
    {"K", "Potassium", 19},
    {"CE", "Cerium", 58},
    {"PR", "Praseodymium", 59},
    {"CS", "Cesium", 55},
    {"PM", "Promethium", 61},
    {"CL", "Chlorine", 17},
    {"PA", "Protactinium", 91},
    {"CR", "Chromium", 24},
    {"RA", "Radium", 88},
    {"CO", "Cobalt", 27},
    {"RN", "Radon", 86},
    {"CU", "Copper", 29},
    {"RE", "Rhenium", 75},
    {"CM", "Curium", 96},
    {"RH", "Rhodium", 45},
    {"DY", "Dysprosium", 66},
    {"RB", "Rubidium", 37},
    {"ES", "Einsteinium", 99},
    {"RU", "Ruthenium", 44},
    {"ER", "Erbium", 68},
    {"RF", "Rutherfordium", 104},
    {"EU", "Europium", 63},
    {"SM", "Samarium", 62},
    {"FM", "Fermium", 100},
    {"SC", "Scandium", 21},
    {"F", "Fluorine", 9},
    {"SG", "Seaborgium", 106},
    {"FR", "Francium", 87},
    {"SE", "Selenium", 34},
    {"GD", "Gadolinium", 64},
    {"SI", "Silicon", 14},
    {"GA", "Gallium", 31},
    {"AG", "Silver", 47},
    {"GE", "Germanium", 32},
    {"NA", "Sodium", 11},
    {"AU", "Gold", 79},
    {"SR", "Strontium", 38},
    {"HF", "Hafnium", 72},
    {"S", "Sulfur", 16},
    {"HA", "Hahnium", 105},
    {"TA", "Tantalum", 73},
    {"HS", "Hassium", 108},
    {"TC", "Technetium", 43},
    {"HE", "Helium", 2},
    {"TE", "Tellurium", 52},
    {"HO", "Holmium", 67},
    {"TB", "Terbium", 65},
    {"H", "Hydrogen", 1},
    {"TL", "Thalium", 81},
    {"IN", "Indium", 49},
    {"TH", "Thorium", 90},
    {"I", "Iodine", 53},
    {"TM", "Thulium", 69},
    {"IR", "Iridium", 77},
    {"SN", "Tin", 50},
    {"FE", "Iron", 26},
    {"TI", "Titanium", 22},
    {"KR", "Krypton", 36},
    {"W", "Tungsten", 74},
    {"LA", "Lanthanum", 57},
    {"U", "Uranium", 92},
    {"LR", "Lawrencium", 103},
    {"V", "Vanadium", 23},
    {"PB", "Lead", 82},
    {"XE", "Xenon", 54},
    {"LI", "Lithium", 3},
    {"YB", "Ytterbium", 70},
    {"LU", "Lutetium", 71},
    {"Y", "Yttrium", 39},
    {"MG", "Magnesium", 12},
    {"ZN", "Zinc", 30},
    {"MN", "Manganese", 25},
    {"ZR", "Zirconium", 40},
    {"MT", "Meitnerium", 109},
  };
  // Van Der Waals radii from
  // http://www.ccdc.cam.ac.uk/support/csd_doc/volume1/z1c07076.html as of 26
  // Feb 2000.  They cite A. Bondi (1964) "van der Waals Volumes and Radii"
  // J.Phys.Chem. 68, 441-451.  Units are Angstroms.
  struct VDWRadius {
    const char *symbol;
    Float radius;
  };
  const VDWRadius VDWRadii [] = {
    {"AG", 1.72}, {"AR", 1.88}, {"AS", 1.85}, {"AU", 1.66}, 
    {"BR", 1.85}, {"C", 1.70}, {"CD", 1.58}, {"CL", 1.75}, 
    {"CU", 1.40}, {"F", 1.47}, {"GA", 1.87}, {"H", 1.20}, 
    {"HE", 1.40}, {"HG", 1.55}, {"I", 1.98}, {"IN", 1.93}, 
    {"K", 2.75}, {"KR", 2.02}, {"LI", 1.82}, {"MG", 1.73}, 
    {"N", 1.55}, {"NA", 2.27}, {"NE", 1.54}, {"NI", 1.63}, 
    {"O", 1.52}, {"P", 1.80}, {"PB", 2.02}, {"PD", 1.63}, 
    {"PT", 1.72}, {"S", 1.80}, {"SE", 1.90}, {"SI", 2.10}, 
    {"SN", 2.17}, {"TE", 2.06}, {"TL", 1.96}, {"U", 1.86}, 
    {"XE", 2.16}, {"ZN", 1.39}, 
  };
  const Float defaultVDWRadius = 2.0;
  struct ElementInfo {
    const char *m_symbol;
    const char *m_name;
    Color m_color;
    Float m_radius;
    ElementInfo ()
      : m_symbol (0)
    {}
  };
  ElementInfo elements [(sizeof (tmpElements) / sizeof (TmpElement)) + 1];
}

// One more than the largest atomic number we know.
const int AtomInfo::maxNumber = sizeof (elements) / sizeof (ElementInfo);

namespace {
  // Next one converts the miscellaneous information from other sources into
  // the simpler "elements" array, and does some consistency checking.
  void fillInElements () {
    for (int i = 0; i < AtomInfo::maxNumber; i++) {
      assert (0 == elements [i].m_symbol);
      elements [i].m_radius = defaultVDWRadius;
      elements [i].m_color = allOtherColor;
    }
    for (unsigned int i = 0; i < sizeof(tmpElements)/sizeof(TmpElement); i++) {
      const TmpElement &te = tmpElements [i];
      const int elemNo = te.number;
      assert (0 == elements [elemNo].m_symbol);
      elements [elemNo].m_symbol = te.symbol;
      elements [elemNo].m_name = te.name;
    }
    for (int i = 1; i < AtomInfo::maxNumber; i++) {
      assert (0 != elements [i].m_symbol);
      char sym [3];
      sym [0] = elements [i].m_symbol [0];
      assert (sym [0]);
      if (elements [i].m_symbol [1]) {
	sym [1] = elements [i].m_symbol [1];
      } else {
	sym [1] = ' ';
      }	
      sym [2] = '\0';
      assert (2 == strlen (sym));
      for (unsigned int j = 0; j < sizeof(colors) / sizeof (AtomColor); j++) {
	const AtomColor &ac = colors [j];
	const char *const listEnd = ac.elements + strlen (ac.elements);
	assert (0 == strlen (ac.elements) % 2);
	for (const char * k = ac.elements; k < listEnd; k += 2) {
	  if (k [0] == sym [0] && k [1] == sym [1]) {
	    elements [i].m_color = hexToColor (ac.rgb);
	  }
	}
      }
      for (unsigned int j = 0; j < sizeof(VDWRadii)/sizeof(VDWRadius); j++) {
	if (0 == strcmp (VDWRadii [j].symbol, elements [i].m_symbol)) {
	  elements [i].m_radius = VDWRadii[j].radius;
	}
      }
    }
  };
  bool useless = (fillInElements (), true);
}

namespace {
  // Check that an atomic number is in bounds.
  inline void checkNumber (int number) {
    (void) number;
    assert (number >= 1 && number < AtomInfo::maxNumber);
  }
}

int AtomInfo::symbolToNumber (const String &symbol) {
  const String ucSym = symbol.upcase ();
  for (int i = 1; i < maxNumber; i++) {
    if (ucSym == elements [i].m_symbol) {
      return i;
    }
  }
  return -1;
}

int AtomInfo::nameToNumber (const String &name) {
  const String ucName = name.upcase ();
  for (int i = 1; i < maxNumber; i++) {
    if (ucName == elements [i].m_name) {
      return i;
    }
  }
  return -1;
}

String AtomInfo::toSymbol (int number) {
  checkNumber (number);
  return elements [number].m_symbol;
}

String AtomInfo::toName (int number) {
  checkNumber (number);
  return elements [number].m_name;
}

Float AtomInfo::VDWRadius (int number) {
  checkNumber (number);
  return elements[number].m_radius;
}

Color AtomInfo::CPKColor (int number) {
  checkNumber (number);
  return elements [number].m_color;
}
