// Fungimol - an extensible system for designing atomic-scale objects.
// Copyright (C) 2000 Tim Freeman
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Library General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
// 
// You should have received a copy of the GNU Library General Public
// License along with this library in the file COPYING.txt; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place -
// Suite 330, Boston, MA 02111-1307, USA
//
// The author can be reached by email at tim@infoscreen.com, or by
// paper mail at:
//
// Tim Freeman
// 655 S. FairOaks Ave., Apt B-316
// Sunnyvale, CA 94086
//

#ifndef __AtomConfiguration_h__
#include "AtomConfiguration.h"
#endif

#ifndef __String_h__
#include "String.h"
#endif

#ifndef __MemoryUtil_h__
#include "MemoryUtil.h"
#endif

#ifndef __PositiveIntSlotValue_h__
#include "PositiveIntSlotValue.h"
#endif

#ifndef __AtomInfo_h__
#include "AtomInfo.h"
#endif

#ifndef __MenuSlotValueChoices_h__
#include "MenuSlotValueChoices.h"
#endif

namespace {
  String symbolNumberName () {
    return "Atomic Number";
  }
  // The list of available drawers.
  // The following idiom allows getMSVC to correctly return the value for msvc
  // even before the static initializers are finished, and that we'll create it
  // only once.
  MenuSlotValueChoices *getMSVC ();
  MenuSlotValueChoices *msvc = getMSVC ();
  MenuSlotValueChoices *getMSVC () {
    if (!msvc) {
      msvc = NEW (MenuSlotValueChoices ());
      msvc->ref ();
    }
    return msvc;
  }
  
  void deallocate () {
    msvc->deref ();
    assert (0 == msvc->count ());
    delete msvc;
    msvc = 0;
  }
  bool useless = (MemoryUtil::registerDeallocator (deallocate), true);
}

AtomConfiguration::AtomConfiguration ()
  : DDPOConfiguration (getMSVC())
{
  // If nobody has called registerDrawer ("SpaceFill", whatever) before we get
  // to the next line, then the next line will abort in debug code.  This is as
  // it should be.
  ref ();
  setDrawerName ("SpaceFill");
  initializeSlot (symbolNumberName (), NEW (PositiveIntSlotValue (6)));
  deref ();
}

int AtomConfiguration::getSymbolNumber () const {
  return PositiveIntSlotValue::getSlot (this, symbolNumberName());
}

void AtomConfiguration::setSymbolNumber (int s) {
  PositiveIntSlotValue::setSlot (this, symbolNumberName(), s);
}

String AtomConfiguration::unParse () const {
  return AtomInfo::toSymbol (getSymbolNumber())+","+getDrawerName ();
}

int AtomConfiguration::getDifferentAtoms () const {
  return AtomInfo::maxNumber * DDPOConfiguration::numAlternatives ();
}

int AtomConfiguration::myArrayPos () const {
  return getSymbolNumber () * DDPOConfiguration::numAlternatives ()
    + DDPOConfiguration::currentDrawerPos ();
}

int AtomConfiguration::generation () const {
  // The only thing that forces discarding caching is the DDPOConfiguration's
  // generation change.
  return DDPOConfiguration::generation ();
}

SP<Configuration> AtomConfiguration::copy () const {
  // Default copy constructor is good.
  return NEW (AtomConfiguration (*this));
}

bool AtomConfiguration::hasTheSameTypeAs (const Configuration *c) const {
  return 0 != dynamic_cast <const AtomConfiguration *> (c);
}

void AtomConfiguration::registerDrawer (const String &shortName,
					const String &prettyName) {
  getMSVC()->addChoice (shortName, prettyName);
}
